<?php

namespace AlloCine\UnifiedTag\Bundle\Worker;

use AlloCine\UnifiedTag\Bundle\Exceptions\UnRecognisedTagException;
use Psr\Cache\InvalidArgumentException;
use Symfony\Component\Cache\Adapter\PhpArrayAdapter;

class UnifiedTagWorker
{
    const TAG3D_FROM_ID = 0;
    const TAG_GRAPH_FROM_ID = 1;
    const ID_FROM_TAG3D = 2;
    const TAG_GRAPH_FROM_TAG3D = 3;
    const ID_FROM_TAG_GRAPH = 4;
    const TAG3D_FROM_TAG_GRAPH = 5;

    /**
     * @var PhpArrayAdapter
     */
    private $warmup;

    public function __construct(PhpArrayAdapter $warmup)
    {
        $this->warmup = $warmup;
    }

    /**
     * @param int $id
     *
     * @return string
     * @throws UnRecognisedTagException|\Exception|InvalidArgumentException
     */
    public function getTag3dFromId(int $id): string
    {
        return $this->getItem((string) $id, self::TAG3D_FROM_ID);
    }

    /**
     * @param int $id
     *
     * @return string
     * @throws UnRecognisedTagException|\Exception|InvalidArgumentException
     */
    public function getTagGraphFromId(int $id): string
    {
        return $this->getItem((string) $id, self::TAG_GRAPH_FROM_ID);
    }

    /**
     * @param string $tag3d
     *
     * @return int
     * @throws UnRecognisedTagException|\Exception|InvalidArgumentException
     */
    public function getIdFromTag3d(string $tag3d): int
    {
        return $this->getItem($tag3d, self::ID_FROM_TAG3D);
    }

    /**
     * @param string $tag3d
     *
     * @return string
     * @throws UnRecognisedTagException|\Exception|InvalidArgumentException
     */
    public function getTagGraphFromTag3d(string $tag3d): string
    {
        return $this->getItem($tag3d, self::TAG_GRAPH_FROM_TAG3D);
    }

    /**
     * @param string $tagGraph
     *
     * @return int
     * @throws UnRecognisedTagException|\Exception|InvalidArgumentException
     */
    public function getIdFromTagGraph(string $tagGraph): int
    {
        return $this->getItem($tagGraph, self::ID_FROM_TAG_GRAPH);
    }

    /**
     * @param string $tagGraph
     *
     * @return string
     * @throws UnRecognisedTagException|\Exception|InvalidArgumentException
     */
    public function getTag3dFromTagGraph(string $tagGraph): string
    {
        return $this->getItem($tagGraph, self::TAG3D_FROM_TAG_GRAPH);
    }

    /**
     * @param string $item
     * @param int    $type
     *
     * @return mixed
     * @throws UnRecognisedTagException|\Exception|InvalidArgumentException
     */
    private function getItem(string $item, int $type)
    {
        switch ($type) {
            case self::TAG3D_FROM_ID:
                $message = sprintf('The tag id "%s" is not found', $item);
                $key = 'tag_3d';
                break;
            case self::TAG_GRAPH_FROM_ID:
                $message = sprintf('The tag id "%s" is not found', $item);
                $key = 'tag_graph';
                break;
            case self::ID_FROM_TAG3D:
                $message = sprintf('The tag3d "%s" is not found. No id to return', $item);
                $key = 'id';
                break;
            case self::TAG_GRAPH_FROM_TAG3D:
                $message = sprintf('The tag3d "%s" is not found. No id to return', $item);
                $key = 'tag_graph';
                break;
            case self::ID_FROM_TAG_GRAPH:
                $message = sprintf('The tag graph "%s" is not found. No id to return', $item);
                $key = 'id';
                break;
            case self::TAG3D_FROM_TAG_GRAPH:
                $message = sprintf('The tag graph "%s" is not found. No id to return', $item);
                $key = 'tag_3d';
                break;
            default:
                throw new \Exception('Unrecognised warmup type.');
        }

        if ($this->warmup->hasItem($item)) {
            return $this->warmup->getItem($item)->get()[$key];
        }

        throw new UnRecognisedTagException($message);
    }
}
