<?php

namespace AlloCine\UnifiedTag\Bundle\Twig\Extension;

use AlloCine\UnifiedTag\Bundle\Exceptions\UnRecognisedTagException;
use AlloCine\UnifiedTag\Bundle\Worker\UnifiedTagWorker;

class UnifiedTagExtension extends \Twig_Extension
{
    const ID_TO_TAG = 0;
    const ID_TO_TAG3D = 1;
    const TAG_TO_ID = 2;
    const TAG_TO_TAG3D = 3;
    const TAG3D_TO_ID = 4;
    const TAG3D_TO_TAG = 5;

    /**
     * @var UnifiedTagWorker
     */
    private $worker;

    public function __construct(UnifiedTagWorker $worker)
    {
        $this->worker = $worker;
    }

    /**
     * {@inheritsDoc}
     */
    public function getFilters()
    {
        return [
            new \Twig_SimpleFilter('ut_id_to_tag', [$this, 'UtIdToTag']),
            new \Twig_SimpleFilter('ut_id_to_tag3d', [$this, 'UtIdToTag3d']),
            new \Twig_SimpleFilter('ut_tag_to_id', [$this, 'UtTagToId']),
            new \Twig_SimpleFilter('ut_tag_to_tag3d', [$this, 'UtTagToTag3d']),
            new \Twig_SimpleFilter('ut_tag3d_to_id', [$this, 'UtTag3dToId']),
            new \Twig_SimpleFilter('ut_tag3d_to_tag', [$this, 'UtTag3dToTag']),
        ];
    }

    /**
     * @param int $id
     *
     * @return string
     * @throws UnRecognisedTagException|\Exception
     */
    public function UtIdToTag(int $id): string
    {
        return $this->useWorker($id, self::ID_TO_TAG);
    }

    /**
     * @param int $id
     *
     * @return string
     * @throws UnRecognisedTagException|\Exception
     */
    public function UtIdToTag3d(int $id): string
    {
        return $this->useWorker($id, self::ID_TO_TAG3D);
    }

    /**
     * @param string $tag
     *
     * @return int
     * @throws UnRecognisedTagException|\Exception
     */
    public function UtTagToId(string $tag): int
    {
        return $this->useWorker($tag, self::TAG_TO_ID);
    }

    /**
     * @param string $tag
     *
     * @return string
     * @throws UnRecognisedTagException|\Exception
     */
    public function UtTagToTag3d(string $tag): string
    {
        return $this->useWorker($tag, self::TAG_TO_TAG3D);
    }

    /**
     * @param string $tag3d
     *
     * @return int
     * @throws UnRecognisedTagException|\Exception
     */
    public function UtTag3dToId(string $tag3d): int
    {
        return $this->useWorker($tag3d, self::TAG3D_TO_ID);
    }

    /**
     * @param string $tag3d
     *
     * @return string
     * @throws UnRecognisedTagException|\Exception
     */
    public function UtTag3dToTag(string $tag3d): string
    {
        return $this->useWorker($tag3d, self::TAG3D_TO_TAG);
    }

    /**
     * @param $key
     * @param $type
     *
     * @return int|string
     * @throws UnRecognisedTagException|\Exception
     */
    private function useWorker($key, $type)
    {
        switch ($type) {
            case self::ID_TO_TAG:
                return $this->worker->getTagGraphFromId($key);
            case self::ID_TO_TAG3D:
                return $this->worker->getTag3dFromId($key);
            case self::TAG_TO_ID:
                return $this->worker->getIdFromTagGraph($key);
            case self::TAG_TO_TAG3D:
                return $this->worker->getTag3dFromTagGraph($key);
            case self::TAG3D_TO_ID:
                return $this->worker->getIdFromTag3d($key);
            case self::TAG3D_TO_TAG:
                return $this->worker->getTagGraphFromTag3d($key);
        }

        throw new \Exception('Unrecognised type of worker');
    }
}
