const fs = require('fs');
const path = require('path');
const {
  loadFileAsync,
  parseXliffString,
} = require('./parser');

const CSV_DIRECTORY = path.join(__dirname, '../src/Bundle/Resources/csv/');
const TRANSLATIONS_DIRECTORY = path.join(__dirname, '../src/Bundle/Resources/translations');

const TAGS = {
  ID: 'id',
  TAG3D: 'tag3d',
  TAGGRAPH: 'tagGraph',
};

const TYPES = {
  ACTIVITIES: 'activities',
  CATEGORIES: 'categories',
  GENRES: 'genres',
  STATUS: 'status',
  THEMES: 'themes',
  TYPES: 'types',
  FORMATS: 'formats',
  CERTIFICATES: 'certificates',
};

const LOCALES = ['de_DE', 'en_GB', 'en_US', 'es_ES', 'es_MX', 'fr_FR', 'pt_BR', 'tr_TR'];

class TagHelper {

  get type() {
    return this._type;
  }

  set type(type) {
    this._type = type;
  }

  constructor(type) {
    if (!Object.values(TYPES).includes(type)) throw new Error('Type not found');
    this.type = type;
    this.setFile(this.getFile(`${CSV_DIRECTORY}${type}.csv`));
    this.localizedFiles = {};
  }

  /**
   * push csv file on object
   *
   * @param {array} file
   */
  setFile(file) {
    this.file = file;
  }

  /**
   * get and parse a csv file
   *
   * @param {string} fileSrc
   */
  getFile(fileSrc) {
    const contents = fs.readFileSync(fileSrc, 'utf8');
    const file = contents
      .split('\n')
      .filter(x => x)
      .map(x => {
        const cols = x.split(';');
        return {
          [TAGS.ID]: parseInt(cols[0]),
          [TAGS.TAG3D]: cols[1],
          [TAGS.TAGGRAPH]: cols[2],
        };
      });
    return file;
  }

  /**
   * get item from file
   *
   * @param {string} from column to search
   * @param {string} match value search
   * @param {string} out column out
   */
  getItem(from, match, out) {
    const line = this.file.find(line => line[from] === match);
    if (!line) return undefined;
    return line[out];
  }

  /**
   * get a 3d tag from a id
   *
   * @param {number} id
   */
  getTag3dFromId(id) {
    return this.getItem(TAGS.ID, id, TAGS.TAG3D);
  }

  /**
   * get a tag graph from a id
   *
   * @param {number} id
   */
  getTagGraphFromId(id) {
    return this.getItem(TAGS.ID, id, TAGS.TAGGRAPH);
  }

  /**
   * get a id from tag 3d
   *
   * @param {string} tag3d
   */
  getIdFromTag3d(tag3d) {
    return this.getItem(TAGS.TAG3D, tag3d, TAGS.ID);
  }

  /**
   * get tag graph from tag 3d
   *
   * @param {string} tag3d
   */
  getTagGraphFromTag3d(tag3d) {
    return this.getItem(TAGS.TAG3D, tag3d, TAGS.TAGGRAPH);
  }

  /**
   * get translated tag graph from tag 3d
   *
   * @param {string} tag3d
   * @param {string} locale
   */
  async getTranslationFromTag3d(tag3d, locale) {
    let localizedRes = this.localizedFiles[locale];
    if (!localizedRes && LOCALES.indexOf(locale) > -1) {
      const localeFilePath = path.join(TRANSLATIONS_DIRECTORY, `graph_${this.type}.${locale}.xlf`);
      try {
        const content = await loadFileAsync({ filePath: localeFilePath });
        localizedRes = parseXliffString(content);
        this.localizedFiles[locale] = localizedRes;
      } catch (err) {}
    }
    return localizedRes ? localizedRes[tag3d] : undefined;
  }

  /**
   * get a id from a tag graph
   *
   * @param {string} tagGraph
   */
  getIdFromTagGraph(tagGraph) {
    return this.getItem(TAGS.TAGGRAPH, tagGraph, TAGS.ID);
  }

  /**
   * get a 3d tag from a tag graph
   *
   * @param {string} tagGraph
   */
  getTag3dFromTagGraph(tagGraph) {
    return this.getItem(TAGS.TAGGRAPH, tagGraph, TAGS.TAG3D);
  }
}

TagHelper.TYPES = TYPES;
TagHelper.LOCALES = LOCALES;

module.exports = TagHelper;
