const path = require('path');
const TagHelper = require('../index');
const { parseXliffString } = require('../parser');

describe('tag bundle', function() {
  describe('amenities', () => {
    let tagBundle;
    beforeAll(() => {
      tagBundle = new TagHelper(TagHelper.TYPES.AMENITIES); // Fake type
      // mock fake file
      tagBundle.setFile(
        tagBundle.getFile(path.join(__dirname, 'amenities.mock.csv')),
      );
    });

    it('should be get a tagGraph from a tag 3d', () => {
      const tag3d = tagBundle.getTagGraphFromTag3d(
        'Showtime.Amenity.DolbyAtmos',
      );
      expect(tag3d).toBe('DolbyAtmos');
    });
  });
  describe('activities', () => {
    let tagBundle;
    beforeAll(() => {
      tagBundle = new TagHelper(TagHelper.TYPES.ACTIVITIES); // Fake type
      // mock fake file
      tagBundle.setFile(
        tagBundle.getFile(path.join(__dirname, 'activities.mock.csv')),
      );
    });

    it('should be get a tag3d from activity id', () => {
      const tag3d = tagBundle.getTag3dFromId(15012);
      expect(tag3d).toBe('Company.Activity.FilmExhibitor');
    });

    it('should be get a tagGraph from activity id', () => {
      const tagGraph = tagBundle.getTagGraphFromId(15012);
      expect(tagGraph).toBe('FILM_EXHIBITOR');
    });

    it('should be get a id from activity id', () => {
      const id = tagBundle.getIdFromTag3d('Company.Activity.FilmExhibitor');
      expect(id).toBe(15012);
    });

    it('should be get a tagGraph from a tag 3d', () => {
      const tag3d = tagBundle.getTagGraphFromTag3d(
        'Company.Activity.FilmExhibitor',
      );
      expect(tag3d).toBe('FILM_EXHIBITOR');
    });

    it('should be get a id from tag graph', () => {
      const tag3d = tagBundle.getIdFromTagGraph('FILM_EXHIBITOR');
      expect(tag3d).toBe(15012);
    });

    it('should be get a tag 3d from tag graph', () => {
      const tag3d = tagBundle.getTag3dFromTagGraph('FILM_EXHIBITOR');
      expect(tag3d).toBe('Company.Activity.FilmExhibitor');
    });
  });
  describe('parser', async () => {
    it('throw errors while parsing wrong content', () => {
      expect(() => parseXliffString('')).toThrow();
      expect(() =>
        parseXliffString(`
        {"lol":true}`),
      ).toThrow();
      expect(() =>
        parseXliffString(`
        <?xml version="1.0" encoding="utf-8"?>          
        <xml>`),
      ).toThrow();
      expect(() =>
        parseXliffString(`
        <?xml version="1.0" encoding="utf-8"?>
          <xliff xmlns="urn:oasis:names:tc:xliff:document:1.2" version="1.2">
            <file source-language="en-GB" target-language="en-US" datatype="plaintext" original="file.ext">              
            </file>
          </xliff>
        <xml>`),
      ).toThrow();
      expect(() =>
        parseXliffString(`
        <?xml version="1.0" encoding="utf-8"?>
          <xliff xmlns="urn:oasis:names:tc:xliff:document:1.2" version="1.2">
            <file source-language="en-GB" target-language="en-US" datatype="plaintext" original="file.ext">
              <header>
                <tool tool-id="symfony" tool-name="Symfony"/>
              </header>
              <body>                
              </body>
            </file>
          </xliff>
        <xml>`),
      ).toThrow();
    });
    it('parse xliff-sf compliant string', () => {
      const parsed = parseXliffString(`
        <?xml version="1.0" encoding="utf-8"?>
          <xliff xmlns="urn:oasis:names:tc:xliff:document:1.2" version="1.2">
            <file source-language="en-GB" target-language="en-US" datatype="plaintext" original="file.ext">
              <header>
                <tool tool-id="symfony" tool-name="Symfony"/>
              </header>
              <body>
                <trans-unit id="Rcxgyz0" resname="Release.Certificate.12">
                  <source>Release.Certificate.12</source>
                  <target>Above 12</target>
                </trans-unit>
                <trans-unit id="T3ke9a5" resname="Release.Certificate.15">
                  <source>Release.Certificate.15</source>
                  <target>Above 15</target>
                </trans-unit>
              </body>
            </file>
          </xliff>
        <xml>`);
      expect(parsed).toMatchSnapshot();
    });
    it('parse from loco generated file', () => {
      const parsed = parseXliffString(`
        <?xml version="1.0" encoding="utf-8"?> 
          <!--
           Loco xml export: XLIFF for Symfony
           Project: Unified Tag Bundle            
          -->
          <xliff xmlns="urn:oasis:names:tc:xliff:document:1.2" version="1.2">
            <!--
             Test node comment            
            -->
            <file source-language="en-GB" target-language="en-US" datatype="plaintext" original="file.ext">
              <!--
               Test node comment            
              -->
              <!--
               Test node comment            
              -->
              <header>
                <tool tool-id="symfony" tool-name="Symfony"/>
              </header>
              <body>                      
                <trans-unit id="Rcxgyz0" resname="Release.Certificate.12">
                  <source>Release.Certificate.12</source>
                  <target>12</target>
                </trans-unit>
                <trans-unit id="T3ke9a5" resname="Release.Certificate.15">
                  <source>Release.Certificate.15</source>
                  <target>15</target>
                </trans-unit>
                <trans-unit id="Gy3duMF" resname="Release.Certificate.18">
                  <source>Release.Certificate.18</source>
                  <target>18</target>
                </trans-unit>
                </body>
            </file>
          </xliff>`);
      expect(parsed).toMatchSnapshot();
    });
  });
  describe('certificates', async () => {
    let tagBundle;
    beforeAll(() => {
      tagBundle = new TagHelper(TagHelper.TYPES.CERTIFICATES);
      tagBundle.setFile(
        tagBundle.getFile(path.join(__dirname, 'certificates.mock.csv')),
      );
    });
    it('should get a tagGraph from a tag 3d', () => {
      const tag3d = tagBundle.getTagGraphFromTag3d(
        'Release.Certificate.InterditAuxMoinsDe16AnsAvecAvertissement',
      );
      expect(tag3d).toBe('InterditAuxMoinsDe16AnsAvecAvertissement');
    });
    it('should return nothing when no locale is available', async () => {
      expect(
        await tagBundle.getTranslationFromTag3d(
          'Release.Certificate.taratata',
          'fr_martian',
        ),
      ).toBeUndefined();
    });
    it('should get a localized translation from a tag 3d', async () => {
      const key =
        'Release.Certificate.InterditAuxMoinsDe16AnsAvecAvertissement';
      expect(await tagBundle.getTranslationFromTag3d(key, 'en_US')).toBe(
        'Forbidden below 16 with caution',
      );
      expect(
        await tagBundle.getTranslationFromTag3d(
          'Release.Certificate.InterditAuxMoinsDe12AnsAvecAvertissement',
          'en_US',
        ),
      ).toBe('Forbidden below 12 with caution');
      expect(await tagBundle.getTranslationFromTag3d(key, 'fr_FR')).toBe(
        'Interdit aux moins de 16 ans avec avertissement',
      );
    });
    it('should get a default localized version from a tag 3d', async () => {
      expect(
        await tagBundle.getTranslationFromTag3d(
          'Release.Certificate.InterditAuxMoinsDe12AnsAvecAvertissement',
          'ma_MA',
          { fallbackLocale: 'en_US' },
        ),
      ).toBe('Forbidden below 12 with caution');
    });
  });
});
