<?php

namespace AlloCine\UnifiedTag\Bundle\Cache;

use Symfony\Component\Cache\Adapter\PhpArrayAdapter;
use Symfony\Component\Finder\Finder;
use Symfony\Component\Finder\SplFileInfo;
use Symfony\Component\HttpKernel\CacheWarmer\CacheWarmerInterface;

class UnifiedTagCacheWarmer implements CacheWarmerInterface
{
    const NUMBER_OF_ROWS = 3;
    const ROW_ID = 0;
    const ROW_TAG_3D = 1;
    const ROW_TAG_GRAPH = 2;

    /**
     * @var PhpArrayAdapter
     */
    private $cache;

    /**
     * @var array
     */
    private $sources;

    /**
     * @var array The storage form is [ 'id' => 'My.Tag.3d' ]
     */
    private $storage = [];

    /**
     * @param PhpArrayAdapter $cache
     * @param array           $sources
     */
    public function __construct(PhpArrayAdapter $cache, array $sources)
    {
        $this->cache = $cache;
        $this->sources = $sources;
        $this->init();
    }

    /**
     * Store in cache warmup the array
     */
    private function init(): void
    {
        $finder = new Finder();
        $files = $finder->files()->in($this->sources);

        $errors = [];

        /** @var SplFileInfo $file */
        foreach ($files as $file) {
            foreach (self::getFileRows($file) as $k => $row) {
                try {
                    if (count($row) === self::NUMBER_OF_ROWS) {
                        $tagGraph = $row[self::ROW_TAG_GRAPH];
                        $tag3d = $row[self::ROW_TAG_3D];
                        $id = (int) $row[self::ROW_ID];
                        if (!empty($id)) {
                            $this->insertInStorage((string) $id, [
                                'tag_3d' => $tag3d,
                                'tag_graph' => $tagGraph,
                            ]);
                        }
                        if (!empty($tag3d)) {
                            $this->insertInStorage($tag3d, [
                                'id' => $id,
                                'tag_graph' => $tagGraph,
                            ]);
                        }
                        if (!empty($tagGraph)) {
                            $this->insertInStorage($tagGraph, [
                                'id' => $id,
                                'tag_3d' => $tag3d,
                            ]);
                        }
                    }
                } catch (\Exception $e) {
                    $errors[] = sprintf("%s [%s, line: %d]", $e->getMessage(), $file->getFilename(), $k);
                }
            }
        }

        if ($errors) {
            throw new \Exception(
                "Exceptions thrown when initializing storage:\n".implode("\n", $errors)
            );
        }
    }

    /**
     * @param       $key
     * @param array $values
     *
     * @throws \Exception
     */
    private function insertInStorage($key, array $values): void
    {
        if (array_key_exists($key, $this->storage)) {
            throw new \Exception(sprintf('The key "%s" already exist.', (string) $key));
        }

        $this->storage[$key] = $values;
    }

    /**
     * {@inheritsDoc}
     */
    public function warmUp($cacheDir)
    {
        $this->cache->warmUp($this->storage);
    }

    /**
     * {@inheritsDoc}
     */
    public function isOptional()
    {
        return false;
    }

    /**
     * @param SplFileInfo $file
     *
     * @return array
     */
    public static function getFileRows(SplFileInfo $file): array
    {
        $rows = [];
        if (($handle = fopen($file->getRealPath(), "r")) !== FALSE) {
            $i = 0;
            while (($data = fgetcsv($handle, null, ";")) !== FALSE) {
                $i++;
                $rows[] = $data;
            }
            fclose($handle);
        }

        return $rows;
    }
}
