<?php

namespace AlloCine\UnifiedTag\Bundle\Worker;

use AlloCine\UnifiedTag\Bundle\Cache\UnifiedTagCacheWarmer;
use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\Finder\Finder;
use Symfony\Component\Finder\SplFileInfo;
use Symfony\Component\Translation\Dumper\XliffFileDumper;
use Symfony\Component\Translation\Loader\XliffFileLoader;
use Symfony\Component\Translation\MessageCatalogue;

class TranslationWorker
{
    const INPUT_EXTENSION = '.csv';
    const INPUT_PATH = __DIR__ . DIRECTORY_SEPARATOR . '..' .
                        DIRECTORY_SEPARATOR . 'Resources' .
                        DIRECTORY_SEPARATOR . 'csv';

    const OUTPUT_EXTENSION = '.xlf';
    const OUTPUT_PATH = __DIR__ . DIRECTORY_SEPARATOR . '..' .
                        DIRECTORY_SEPARATOR . 'Resources' .
                        DIRECTORY_SEPARATOR . 'translations';

    const DOMAIN = 'messages';
    const LOCALISATIONS = ['en_GB', 'en_US', 'de_DE', 'es_ES', 'es_MX', 'es_CO', 'fr_FR', 'tr_TR', 'pt_BR'];

    /**
     * @var array
     */
    private $existingMessages = [];

    /**
     * @var array
     */
    private $declaredKeys = [];

    /**
     * @param string $local
     */
    private function storedExistingMessages(string $local): void
    {
        $loader = new XliffFileLoader();
        $finder = new Finder();
        $finder->files()->in(self::OUTPUT_PATH);

        /** @var SplFileInfo $file */
        foreach ($finder as $file) {
            $baseName = $file->getBasename(self::OUTPUT_EXTENSION);
            if (strstr($baseName, $local)) {
                $name = str_replace(['.' . $local, 'graph_'], ['', ''], $baseName);
                $cat = $loader->load($file->getRealPath(), $local);
                $this->existingMessages[$name] = $cat->all(self::DOMAIN);
            }
        }
    }

    private function getAllMessages(): void
    {
        $finder = new Finder();
        $finder->files()->in(self::INPUT_PATH);

        /** @var SplFileInfo $file */
        foreach ($finder as $file) {
            $name = $file->getBasename(self::INPUT_EXTENSION);
            $rows = UnifiedTagCacheWarmer::getFileRows($file);
            array_map(function ($row) use ($name) {
                if (count($row) === UnifiedTagCacheWarmer::NUMBER_OF_ROWS) {
                    if (!empty($row[UnifiedTagCacheWarmer::ROW_TAG_3D])) {
                        $this->declaredKeys[$name][] = $row[UnifiedTagCacheWarmer::ROW_TAG_3D];
                    }
                }
            }, $rows);
        }
    }

    private function addNewTranslations(): void
    {
        foreach ($this->declaredKeys as $file => $keys) {
            foreach ($keys as $key) {
                if (empty($this->existingMessages[$file]) || !array_key_exists($key, $this->existingMessages[$file])) {
                    $this->existingMessages[$file][$key] = 'missing__' . $key;
                }
            }
        }
    }

    private function purgeNotExistingKeys(): void
    {
        $tmpArray = $this->existingMessages;
        foreach ($tmpArray as $file => $keys) {
            foreach ($keys as $key => $value) {
                if (!in_array($key, $this->declaredKeys[$file])) {
                    unset($this->existingMessages[$file][$key]);
                }
            }
        }
    }

    /**
     * @param string $local
     */
    private function writeFile(string $local): void
    {
        $fileSystem = new Filesystem();
        foreach ($this->existingMessages as $file => $keys) {
            $fileName = sprintf('graph_%s.%s%s', $file, $local, self::OUTPUT_EXTENSION);
            $catalogue = new MessageCatalogue($local);
            $catalogue->add($keys, self::DOMAIN);

            $dumper = new XliffFileDumper();
            $content = $dumper->formatCatalogue($catalogue, self::DOMAIN, [
                'default_locale' => 'en_GB',
                'xliff_version' => '1.2'
            ]);

            $path = self::OUTPUT_PATH . DIRECTORY_SEPARATOR . $fileName;
            if ($fileSystem->exists($path)) {
                $fileSystem->remove($path);
            }
            file_put_contents($path, $content);
        }
    }

    /**
     * @param string $local
     */
    private function generateForLocal(string $local): void
    {
        $this->storedExistingMessages($local);
        $this->getAllMessages();
        $this->addNewTranslations();
        $this->purgeNotExistingKeys();
        $this->writeFile($local);
    }

    public function generateFiles(): void
    {
        foreach (self::LOCALISATIONS as $local) {
            $this->generateForLocal($local);
        }
    }
}
