<?php

declare(strict_types=1);

namespace AlloCine\UnifiedTag\Bundle\Twig\Extension;

use Override;
use Psr\Cache\InvalidArgumentException;
use AlloCine\UnifiedTag\Bundle\Exceptions\UnRecognisedTagException;
use AlloCine\UnifiedTag\Bundle\Icu\IcuCompatibility;
use AlloCine\UnifiedTag\Bundle\Worker\UnifiedTagWorker;
use Exception;
use Twig\Extension\AbstractExtension;
use Twig\TwigFilter;

class UnifiedTagExtension extends AbstractExtension
{
    public function __construct(private readonly UnifiedTagWorker $worker)
    {
    }

    #[Override]
    public function getFilters(): array
    {
        return [
            new TwigFilter('ut_id_to_tag', $this->UtIdToTag(...)),
            new TwigFilter('ut_id_to_tag3d', $this->UtIdToTag3d(...)),
            new TwigFilter('ut_tag_to_id', $this->UtTagToId(...)),
            new TwigFilter('ut_tag_to_tag3d', $this->UtTagToTag3d(...)),
            new TwigFilter('ut_tag3d_to_id', $this->UtTag3dToId(...)),
            new TwigFilter('ut_tag3d_to_tag', $this->UtTag3dToTag(...)),
            new TwigFilter('ut_icu_gender_compat', IcuCompatibility::genderConversion(...)),
        ];
    }

    /**
     * @throws UnRecognisedTagException|Exception|InvalidArgumentException
     */
    public function UtIdToTag(int $id): string
    {
        return $this->worker->getTagGraphFromId($id);
    }

    /**
     * @throws UnRecognisedTagException|Exception|InvalidArgumentException
     */
    public function UtIdToTag3d(int $id): string
    {
        return $this->worker->getTag3dFromId($id);
    }

    /**
     * @throws UnRecognisedTagException|Exception|InvalidArgumentException
     */
    public function UtTagToId(string $tag): int
    {
        return $this->worker->getIdFromTagGraph($tag);
    }

    /**
     * @throws UnRecognisedTagException|Exception|InvalidArgumentException
     */
    public function UtTagToTag3d(?string $tag): string
    {
        if ($tag === null || $tag === '' || $tag === '0') {
            return '';
        }

        return $this->worker->getTag3dFromTagGraph($tag);
    }

    /**
     * @throws UnRecognisedTagException|Exception|InvalidArgumentException
     */
    public function UtTag3dToId(string $tag3d): int
    {
        return $this->worker->getIdFromTag3d($tag3d);
    }

    /**
     * @throws UnRecognisedTagException|Exception|InvalidArgumentException
     */
    public function UtTag3dToTag(?string $tag3d): string
    {
        if ($tag3d === null || $tag3d === '' || $tag3d === '0') {
            return '';
        }

        return $this->worker->getTagGraphFromTag3d($tag3d);
    }
}
