<?php

declare(strict_types=1);

namespace AlloCine\UnifiedTag\Bundle\Cache;

use Exception;
use RuntimeException;
use Symfony\Component\Cache\Adapter\PhpArrayAdapter;
use Symfony\Component\Finder\Finder;
use Symfony\Component\Finder\SplFileInfo;
use Symfony\Component\HttpKernel\CacheWarmer\CacheWarmerInterface;

class UnifiedTagCacheWarmer implements CacheWarmerInterface
{
    const int NUMBER_OF_ROWS = 3;
    const int ROW_ID = 0;
    const int ROW_TAG_3D = 1;
    const int ROW_TAG_GRAPH = 2;

    /**
     * The storage format is [ 'id' => 'My.Tag.3d' ]
     */
    private array $storage = [];

    public function __construct(private readonly PhpArrayAdapter $cache, private readonly array $sources)
    {
        $this->init();
    }

    /**
     * Store in cache warmup the array
     */
    private function init(): void
    {
        $finder = new Finder();
        $files = $finder->files()->in($this->sources);

        $errors = [];

        /** @var SplFileInfo $file */
        foreach ($files as $file) {
            foreach (self::getFileRows($file) as $k => $row) {
                try {
                    if (count($row) === self::NUMBER_OF_ROWS) {
                        $tagGraph = $row[self::ROW_TAG_GRAPH];
                        $tag3d = $row[self::ROW_TAG_3D];
                        $id = (int) $row[self::ROW_ID];
                        if ($id !== 0) {
                            $this->insertInStorage((string) $id, [
                                'tag_3d' => $tag3d,
                                'tag_graph' => $tagGraph,
                            ]);
                        }
                        if (!empty($tag3d)) {
                            $this->insertInStorage($tag3d, [
                                'id' => $id,
                                'tag_graph' => $tagGraph,
                            ]);
                        }
                        if (!empty($tagGraph)) {
                            $this->insertInStorage($tagGraph, [
                                'id' => $id,
                                'tag_3d' => $tag3d,
                            ]);
                        }
                    }
                } catch (Exception $e) {
                    $errors[] = sprintf("%s [%s, line: %d]", $e->getMessage(), $file->getFilename(), $k);
                }
            }
        }

        if ($errors !== []) {
            throw new RuntimeException(
                "Exceptions thrown when initializing storage:\n".implode("\n", $errors)
            );
        }
    }

    private function insertInStorage(mixed $key, array $values): void
    {
        if (array_key_exists($key, $this->storage)) {
            throw new RuntimeException(sprintf('The key "%s" already exist.', $key));
        }

        $this->storage[$key] = $values;
    }

    public function warmUp(string $cacheDir, ?string $buildDir = null): array
    {
        return $this->cache->warmUp($this->storage);
    }

    public function isOptional(): bool
    {
        return false;
    }

    public static function getFileRows(SplFileInfo $file): array
    {
        $rows = [];
        if (($handle = fopen($file->getRealPath(), 'rb')) !== false) {
            while (($data = fgetcsv($handle, null, ";", '"', '\\')) !== false) {
                $rows[] = $data;
            }

            fclose($handle);
        }

        return $rows;
    }
}
