import { readdirSync } from 'node:fs'
import { readFile, readdir } from 'node:fs/promises'
import path from 'node:path'

import singletonAsync from '@grainrigi/async-singleton'

import { parseXliffString } from './parser'

const FILE_OPTS = {
	encoding: 'utf8',
} as const

// retrieve translations from php project
const CSV_DIRECTORY = path.join(__dirname, '../../src/Bundle/Resources/csv/')
const TRANSLATIONS_DIRECTORY = path.join(__dirname, '../../src/Bundle/Resources/translations')
const ID = 'id' as const
const TAG3D = 'tag3d' as const
const TAGGRAPH = 'tagGraph' as const
export const CATEGORIES_KEYS = readdirSync(CSV_DIRECTORY, FILE_OPTS)
	?.filter(fname => fname.endsWith('.csv'))
	.reduce((a, i) => {
		const category = path.basename(i, '.csv')
		a[category.toUpperCase()] = category
		return a
	}, {} as Record<string, string>)

export const CATEGORIES = Object.values(CATEGORIES_KEYS)

export enum LocalesEnum {
	'de_DE' = 'de_DE',
	'en_GB' = 'en_GB',
	'en_US' = 'en_US',
	'es_CO' = 'es_CO',
	'es_ES' = 'es_ES',
	'es_MX' = 'es_MX',
	'fr_FR' = 'fr_FR',
	'pt_BR' = 'pt_BR',
	'tr_TR' = 'tr_TR',
}
export type Locales = LocalesEnum | keyof typeof LocalesEnum
const LOCALES = Object.values(LocalesEnum)

export type UTBTabStruct = {
	[ID]: number
	[TAG3D]: string
	[TAGGRAPH]: string
	category: string
} & {
	[L in LocalesEnum]?: string
}

let initialized = false
export const tag3dIndex = new Map<string, UTBTabStruct>()
export const idIndex = new Map<number, UTBTabStruct>()
export const tagGraphIndex = new Map<string, UTBTabStruct>()

/**
 * Initialize and populates the helper from csv and xliff files
 */
export const init = singletonAsync(async () => {
	if (initialized) return

	const csvFiles = (await readdir(CSV_DIRECTORY, FILE_OPTS)).filter(f => f.match(/.csv$/i))
	// Index everything by tag3d
	for (const csvFile of csvFiles) {
		const key = path.basename(csvFile, '.csv')
		const csvFileContent = await readFile(path.resolve(CSV_DIRECTORY, csvFile), FILE_OPTS)
		const csvLines = csvFileContent.split('\n')
		// index tag3ds from each file
		for (const csvLine of csvLines) {
			if (!csvLine) continue
			const [id, tag3d, tagGraph] = csvLine.split(';')
			tag3dIndex.set(
				tag3d,
				tag3dIndex.get(tag3d) ?? {
					[ID]: +id,
					[TAG3D]: tag3d,
					[TAGGRAPH]: tagGraph,
					'category': key,
				},
			)
		}
		// augment everything with related translations
		for (const locale of LOCALES) {
			try {
				const content = await readFile(path.resolve(TRANSLATIONS_DIRECTORY, `graph_${key}.${locale}.xlf`), FILE_OPTS)
				const localizedRes = parseXliffString(content)
				for (const [tag3d, trans] of localizedRes) {
					if (!tag3dIndex.has(tag3d)) continue
					;(tag3dIndex.get(tag3d) as UTBTabStruct)[locale] = trans
				}
				// eslint-disable-next-line no-empty
			} catch (err) {}
		}
	}
	// index by if or tag graph if present in files
	for (const [, val] of tag3dIndex) {
		if (val[ID] && !!val[ID]) idIndex.set(val[ID], val)
		if (val[TAGGRAPH] && !!val[TAGGRAPH]) tagGraphIndex.set(val[TAGGRAPH], val)
	}
	initialized = true
})

export const getTag3dFromId = (id: number) => init().then(() => idIndex.get(id)?.[TAG3D])

export const getTagGraphFromId = (id: number) => init().then(() => idIndex.get(id)?.[TAGGRAPH])

export const getIdFromTag3d = (tag3d: string) => init().then(() => tag3dIndex.get(tag3d)?.[ID])

export const getTagGraphFromTag3d = (tag3d: string) => init().then(() => tag3dIndex.get(tag3d)?.[TAGGRAPH])

/**
 * Get translated tag graph from tag 3d
 */
export const getTranslationFromTag3d = async (
	tag3d: string,
	locale: string,
	options: { fallbackLocale?: Locales } = { fallbackLocale: undefined },
) => {
	await init()
	let ret
	const { fallbackLocale } = options
	if (!tag3dIndex.has(tag3d)) return ret
	const item = tag3dIndex.get(tag3d) as UTBTabStruct
	if (locale in item) ret = item[locale as LocalesEnum]
	if (!ret && fallbackLocale) ret = item[fallbackLocale]
	return ret
}

export const getIdFromTagGraph = (tagGraph: string) => init().then(() => tagGraphIndex.get(tagGraph)?.[ID])

export const getTag3dFromTagGraph = (tagGraph: string) => init().then(() => tagGraphIndex.get(tagGraph)?.[TAG3D])

/**
 * Get all tags of a category
 */
export const getAllFromCategory = async (category: string) => {
	await init()
	const ret = new Map<string, UTBTabStruct>()
	for (const [tag3d, item] of tag3dIndex) {
		if (item.category === category) ret.set(tag3d, item)
	}
	return ret
}
