import {
	CATEGORIES,
	getAllFromCategory,
	getIdFromTag3d,
	getIdFromTagGraph,
	getTag3dFromId,
	getTag3dFromTagGraph,
	getTagGraphFromId,
	getTagGraphFromTag3d,
	getTranslationFromTag3d,
	idIndex,
	init,
	tag3dIndex,
	tagGraphIndex,
} from '../index'
import { parseXliffString } from '../parser'

describe('unified tag bundle', () => {
	beforeAll(async () => {
		await init()
	})
	it('idIndex', async () => {
		expect(idIndex).toMatchSnapshot()
	})
	it('tag3dIndex', async () => {
		expect(tag3dIndex).toMatchSnapshot()
	})
	it('tagGraphIndex', async () => {
		expect(tagGraphIndex).toMatchSnapshot()
	})
	it('tagGraph from a tag 3d', async () => {
		expect(await getTagGraphFromTag3d('Auditorium.Experience.DolbyAtmos')).toMatchSnapshot()
	})
	it('tag3d from activity id', async () => {
		expect(await getTag3dFromId(15012)).toBe('Activity.Company.FilmExhibitor')
	})
	it('tagGraph from activity id', async () => {
		expect(await getTagGraphFromId(15012)).toBe('FILM_EXHIBITOR')
	})
	it('id from activity id', async () => {
		expect(await getIdFromTag3d('Activity.Company.FilmExhibitor')).toBe(15012)
	})
	it('tagGraph from a tag 3d', async () => {
		expect(await getTagGraphFromTag3d('Activity.Company.FilmExhibitor')).toBe('FILM_EXHIBITOR')
	})
	it('id from tag graph', async () => {
		expect(await getIdFromTagGraph('FILM_EXHIBITOR')).toBe(15012)
	})
	it('tag 3d from tag graph', async () => {
		expect(await getTag3dFromTagGraph('FILM_EXHIBITOR')).toBe('Activity.Company.FilmExhibitor')
	})
	describe('parseXliffString', () => {
		it('throw errors while parsing wrong content', () => {
			expect(() => parseXliffString('')).toThrow()
			expect(() =>
				parseXliffString(`
        {"lol":true}`),
			).toThrow()
			expect(() =>
				parseXliffString(`
        <?xml version="1.0" encoding="utf-8"?>
        <xml>`),
			).toThrow()
			expect(() =>
				parseXliffString(`
        <?xml version="1.0" encoding="utf-8"?>
          <xliff xmlns="urn:oasis:names:tc:xliff:document:1.2" version="1.2">
            <file source-language="en-GB" target-language="en-US" datatype="plaintext" original="file.ext">
            </file>
          </xliff>
        <xml>`),
			).toThrow()
			expect(() =>
				parseXliffString(`
        <?xml version="1.0" encoding="utf-8"?>
          <xliff xmlns="urn:oasis:names:tc:xliff:document:1.2" version="1.2">
            <file source-language="en-GB" target-language="en-US" datatype="plaintext" original="file.ext">
              <header>
                <tool tool-id="symfony" tool-name="Symfony"/>
              </header>
              <body>
              </body>
            </file>
          </xliff>
        <xml>`),
			).toThrow()
		})
		it('parse xliff-sf compliant string', () => {
			const parsed = parseXliffString(`
        <?xml version="1.0" encoding="utf-8"?>
          <xliff xmlns="urn:oasis:names:tc:xliff:document:1.2" version="1.2">
            <file source-language="en-GB" target-language="en-US" datatype="plaintext" original="file.ext">
              <header>
                <tool tool-id="symfony" tool-name="Symfony"/>
              </header>
              <body>
                <trans-unit id="Rcxgyz0" resname="Release.Certificate.12">
                  <source>Release.Certificate.12</source>
                  <target>Above 12</target>
                </trans-unit>
                <trans-unit id="T3ke9a5" resname="Release.Certificate.15">
                  <source>Release.Certificate.15</source>
                  <target>Above 15</target>
                </trans-unit>
              </body>
            </file>
          </xliff>
        <xml>`)
			expect(parsed).toMatchSnapshot()
		})
		it('parse from loco generated file', () => {
			const parsed = parseXliffString(`
        <?xml version="1.0" encoding="utf-8"?>
          <!--
           Loco xml export: XLIFF for Symfony
           Project: Unified Tag Bundle
          -->
          <xliff xmlns="urn:oasis:names:tc:xliff:document:1.2" version="1.2">
            <!--
             Test node comment
            -->
            <file source-language="en-GB" target-language="en-US" datatype="plaintext" original="file.ext">
              <!--
               Test node comment
              -->
              <!--
               Test node comment
              -->
              <header>
                <tool tool-id="symfony" tool-name="Symfony"/>
              </header>
              <body>
                <trans-unit id="Rcxgyz0" resname="Release.Certificate.12">
                  <source>Release.Certificate.12</source>
                  <target>12</target>
                </trans-unit>
                <trans-unit id="T3ke9a5" resname="Release.Certificate.15">
                  <source>Release.Certificate.15</source>
                  <target>15</target>
                </trans-unit>
                <trans-unit id="Gy3duMF" resname="Release.Certificate.18">
                  <source>Release.Certificate.18</source>
                  <target>18</target>
                </trans-unit>
                </body>
            </file>
          </xliff>`)
			expect(parsed).toMatchSnapshot()
		})

		it('parse new sf generated file (without resname)', () => {
			const parsed = parseXliffString(`
      <xliff xmlns="urn:oasis:names:tc:xliff:document:1.2" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" version="1.2" xsi:schemaLocation="urn:oasis:names:tc:xliff:document:1.2 http://docs.oasis-open.org/xliff/v1.2/os/xliff-core-1.2-strict.xsd">
      <file original="file.ext" source-language="en-GB" target-language="en-GB" datatype="plaintext" tool-id="loco">
        <header>
          <tool tool-id="loco" tool-name="Loco for Symfony" tool-version="1.0.27 20230228-1" tool-company="Loco"/>
        </header>
        <body>
          <trans-unit id="loco:5dada3994d3dd7479a2cac22">
            <source>Activity.Company.Agents</source>
            <target>Agents</target>
          </trans-unit>
          <trans-unit id="loco:5dada3994d3dd7479a2cac23">
            <source>Activity.Company.FilmExhibitor</source>
            <target>Film exhibitor</target>
          </trans-unit>
          <trans-unit id="loco:5dada3994d3dd7479a2cac24">
            <source>Activity.Company.InternationalDistributionExports</source>
            <target>International Distribution/Exports</target>
          </trans-unit>
          <trans-unit id="loco:5dada3994d3dd7479a2cac25">
            <source>Activity.Company.NetworkServiceProvider</source>
            <target>Network service provider</target>
          </trans-unit>
        </body>
      </file>`)
			expect(parsed).toMatchSnapshot()
		})
	})
	describe('getTranslationFromTag3d', () => {
		it('nothing when no locale is available', async () => {
			expect(await getTranslationFromTag3d('Release.Certificate.taratata', 'fr_martian')).toBeUndefined()
		})
		it('localized translation from a tag 3d', async () => {
			const key = 'Release.Certificate.InterditAuxMoinsDe16AnsAvecAvertissement'
			expect(await getTranslationFromTag3d(key, 'en_US')).toBe('Forbidden below 16 with caution')
			expect(await getTranslationFromTag3d('Release.Certificate.InterditAuxMoinsDe12AnsAvecAvertissement', 'en_US')).toBe(
				'Forbidden below 12 with caution',
			)
			expect(await getTranslationFromTag3d(key, 'fr_FR')).toBe('Interdit aux moins de 16 ans avec avertissement')
		})
		it('default localized version from a tag 3d', async () => {
			expect(
				await getTranslationFromTag3d('Release.Certificate.InterditAuxMoinsDe12AnsAvecAvertissement', 'ma_MA', {
					fallbackLocale: 'en_US',
				}),
			).toBe('Forbidden below 12 with caution')
		})
	})
	describe('getAllFromCategory', () => {
		CATEGORIES.forEach(c => {
			it(`category ${c}`, async () => {
				expect(await getAllFromCategory(c)).toMatchSnapshot()
			})
		})
	})
})
