SHELL := /bin/bash
.DEFAULT_GOAL := help
CURRENT_DIR := $(shell dirname $(realpath $(lastword $(MAKEFILE_LIST))))
MAKE_DIR := $(shell dirname $(realpath $(firstword $(MAKEFILE_LIST))))
ifndef ROOT_DIR
ROOT_DIR := $(shell realpath "${CURRENT_DIR}/.." )
endif

GCP_ORG ?= webedia-prod
export WEBEDIA_HOME := ${HOME}/.webedia
export DOCKER_WEBEDIA_HOME := /home/webedia

ADC_FILE := gcloud/${GCP_ORG}/config/application_default_credentials.json
ADC_FILE_HOST := ${WEBEDIA_HOME}/${ADC_FILE}
export ADC_FILE_DOCKER := ${DOCKER_WEBEDIA_HOME}/${ADC_FILE}

define get_ssh_sock
	if [ "$$(uname -s)" = "Darwin" ]; then \
		echo "/run/host-services/ssh-auth.sock"; \
	else \
		if [ -n "$$SSH_AUTH_SOCK" ]; then \
			readlink -f "$$SSH_AUTH_SOCK"; \
		else \
			echo ""; \
		fi \
	fi
endef
export HOST_SSH_AUTH_SOCK := $(shell $(get_ssh_sock))

ifneq (,$(wildcard $(MAKE_DIR)/.env.makefile))
	include $(MAKE_DIR)/.env.makefile
	export $(shell sed 's/=.*//' $(MAKE_DIR)/.env.makefile)
endif

ifeq ($(shell uname -s),Linux)
	export DOCKER_BUILDKIT = 1
endif

export USER_ID ?=$(shell id -u)
export GROUP_ID ?= $(shell id -g)
export APP_ENV ?= dev

DOCKER_RUN = docker run --rm -ti -v ${CURRENT_DIR}/..:/workspace
ifeq (${APP_ENV},dev)
	IGNORE := $(shell mkdir -p "${WEBEDIA_HOME}"/gcloud/${GCP_ORG}/config "${WEBEDIA_HOME}"/.terraform.d/plugin-cache "${WEBEDIA_HOME}"/history "${WEBEDIA_HOME}"/boundary "${WEBEDIA_HOME}"/certs)
	DOCKER_RUN := ${DOCKER_RUN} -v ${WEBEDIA_HOME}:${DOCKER_WEBEDIA_HOME}
endif

ifeq ($(shell test -e ${WEBEDIA_HOME}/gitkeypath && echo -n yes), yes)
	GIT_KEY_PATH = $(shell cat ${WEBEDIA_HOME}/gitkeypath)
	GITLAB_CUSTOM_OPTIONS = -v ${GIT_KEY_PATH}:${GIT_KEY_PATH} -e GIT_SSH_COMMAND="ssh -i ${GIT_KEY_PATH} -o IdentitiesOnly=yes"
endif

COPIER_VOLUMES_ADDS =
ifneq (,$(wildcard ${HOME}/.gitconfig))
	COPIER_VOLUMES_ADDS := ${COPIER_VOLUMES_ADDS} -v ${HOME}/.gitconfig:/var/www/.gitconfig
endif

GIT_IGNORE_GLOBAL :=$(shell git config --global --get core.excludesFile)
ifneq (,$(wildcard ${GIT_IGNORE_GLOBAL}))
GIT_IGNORE_GLOBAL_FULL_PATH :=  $(shell readlink -f ${GIT_IGNORE_GLOBAL})
ifneq (,$(wildcard ${GIT_IGNORE_GLOBAL_FULL_PATH}))
	COPIER_VOLUMES_ADDS := ${COPIER_VOLUMES_ADDS} -v $(shell readlink -f ${GIT_IGNORE_GLOBAL}):$(shell echo '${GIT_IGNORE_GLOBAL}' | sed 's/~/\/var\/www/')
endif
endif

COPIER_VERSION ?= 9.7
COPIER_PREFIX = ${DOCKER_RUN} ${GITLAB_CUSTOM_OPTIONS} \
	-w /workspace \
	-v ${HOME}/.ssh:/var/www/.ssh \
	-v ${HOST_SSH_AUTH_SOCK}:/ssh-agent.sock \
	${COPIER_VOLUMES_ADDS} \
	-e SSH_AUTH_SOCK=/ssh-agent.sock \
	-e USER_ID=${USER_ID} \
	-e GROUP_ID=${GROUP_ID} \
	webedia/copier:${COPIER_VERSION}

## —— Utils 🔧  ———————————————————————————————————————————————————————————————————————
check-command-%:
	@: $(if $(shell which $*),,$(error command not found: $*))

var-%:
	@: $(if $(value $*),,$(error variable $* is undefined))

define color_text
	printf "\e[1;$(2)m$(1)\e[0m\n"
endef

.env: .env.$(APP_ENV).dist
	@test -f $@ \
		&& $(call color_text,/!\ $< might have been modified - remove $@ to be up-to-date,31) \
		|| ( echo "Copy $@ from $<"; cp $< $@)

clean: ## Clean all files not versioned (proceed with caution)
	@$(call color_text,/!\ Warning this will remove all non versionned files. /!\ ,33)
	@$(call color_text,Continue: yes/(no) ? ,37)
	@read doClean && if [ "$$doClean" = "yes" ]; then $(call color_text, Cleaning... ,32); git clean -xdf; fi

.PHONY: forge_recopy
forge_recopy: COPIER_CMD := recopy
forge_recopy: forge_base ## Regenerate forge template files (dryrun=true to simulate, ref=commit to specify a commit). Not recommanded but can resolve conflicts with template evolution.

.PHONY: forge_update
forge_update: COPIER_CMD := update
forge_update: forge_base ## Update forge template files (dryrun=true to simulate, ref=commit to specify a commit)

.PHONY: forge_base
forge_base: COPIER_DRY_RUN := $(if $(filter "${dryrun}","true"),--pretend,)
forge_base: COPIER_REF := $(if ${ref},--vcs-ref=${ref},)
forge_base: COPIER_FILE ?= .forge-answers.yml
forge_base:
ifeq ($(shell git status --porcelain),)
	${COPIER_PREFIX} copier ${COPIER_CMD} -a ${COPIER_FILE} --trust ${COPIER_DRY_RUN} ${COPIER_REF}
else
	@$(call color_text,/!\ Stash all your work before ${COPIER_CMD} template code /!\ ,31)
	git status --porcelain
endif

.PHONY: copier_sh
copier_sh: ## Connect into copier container
	${COPIER_PREFIX} bash

.PHONY: install_git_hooks
install_git_hooks:   ## install git hooks
ifeq ($(APP_ENV),$(filter $(APP_ENV),dev))
	$(MAKE) .git/hooks/prepare-commit-msg
	$(MAKE) .git/hooks/commit-msg
endif

.git/hooks/prepare-commit-msg: .git-hooks/prepare-commit-msg
	cp .git-hooks/prepare-commit-msg .git/hooks/prepare-commit-msg

.git/hooks/commit-msg: .git-hooks/commit-msg
	cp .git-hooks/commit-msg .git/hooks/commit-msg

.PHONY: wbd_check_update
ifeq (${APP_ENV},dev)
wbd_check_update: ${WEBEDIA_HOME}/.last_update
	@cat ${WEBEDIA_HOME}/.last_update | grep -q `date +"%Y-%U"` || $(MAKE) wbd_update_images
else
wbd_check_update:
endif

${WEBEDIA_HOME}/.last_update:
	date +"%Y-%U" >  $@

.SILENT: wbd_update_images
.PHONY: wbd_update_images
wbd_update_images:
	$(call color_text,Verify docker image updates.,37)
	$(call color_text,Continue: (y)/n ? ,37)
	read doUpdate && if [ "$$doUpdate" = "y" ] || [ "$$doUpdate" = "" ]; then \
	$(call color_text, Updating... ,32); \
	$(MAKE) .wbd_download_images; \
	else \
	$(call color_text, Cancel... ,31); \
	fi

.SILENT: .wbd_download_images
.PHONY: .wbd_download_images
.wbd_download_images:
	docker pull europe-docker.pkg.dev/prj-shd-prd-registry-8ed4/foundation/boundary-client:0.19
	docker pull europe-docker.pkg.dev/prj-shd-prd-registry-8ed4/foundation/commitizen:19.3
	docker pull europe-docker.pkg.dev/prj-shd-prd-registry-8ed4/foundation/nginx:1.27-alpine
	docker pull google/cloud-sdk:slim
	docker pull webedia/copier:9.7
	date +"%Y-%U" >  ${WEBEDIA_HOME}/.last_update

## —— The Makefile ℹ️  —————————————————————————————————————————————————————————————————
.PHONY: help
help: ## Outputs this help screen
	@grep -h -E '(^[a-zA-Z0-9_-]+:.*?##.*$$)|(^##)' $(MAKEFILE_LIST) | awk 'BEGIN {FS = ":.*?## "}{printf "\033[32m%-30s\033[0m %s\n", $$1, $$2}' | sed -e 's/\[32m##/[33m/'
