const { readFile, readdir } = require('mz/fs');
const path = require('path');
const { parseXliffString } = require('./parser');

const FILE_OPTS = {
  encoding: 'utf8',
};
const CSV_DIRECTORY = path.join(__dirname, '../src/Bundle/Resources/csv/');
const TRANSLATIONS_DIRECTORY = path.join(
  __dirname,
  '../src/Bundle/Resources/translations',
);
const ID = 'id';
const TAG3D = 'tag3d';
const TAGGRAPH = 'tagGraph';
const CATEGORIES_KEYS = {
  ACTIVITIES: 'activities',
  AMENITIES: 'amenities',
  CARDS: 'cards',
  CATEGORIES: 'categories',
  CERTIFICATES: 'certificates',
  COMFORT: 'comforts',
  EXPERIENCES: 'experiences',
  FORMATS: 'formats',
  GENRES: 'genres',
  STATUS: 'status',
  THEMES: 'themes',
  TYPES: 'types',
};
const CATEGORIES = Object.values(CATEGORIES_KEYS);
const FALLBACK_LOCALE = 'en_US';
const LOCALES = [
  'de_DE',
  'en_GB',
  FALLBACK_LOCALE,
  'es_CO',
  'es_ES',
  'es_MX',
  'fr_FR',
  'pt_BR',
  'tr_TR',
];

let initialized = false;
const tag3dIndex = new Map();
const idIndex = new Map();
const tagGraphIndex = new Map();
/**
 * Initialize and populates the helper from csv and xliff files
 * @returns {Promise<void>}
 */
const init = async () => {
  // Singleton trick for speed
  if (!initialized) {
    // Retrieve all csv files
    const csvFiles = (await readdir(CSV_DIRECTORY, FILE_OPTS)).filter(f =>
      f.match(/.csv$/i),
    );
    // Index everything by tag3d
    for (const csvFile of csvFiles) {
      const key = path.basename(csvFile, '.csv');
      const csvFilePath = path.resolve(CSV_DIRECTORY, csvFile);
      let csvFileContent = await readFile(csvFilePath, FILE_OPTS);
      const csvLines = csvFileContent.split('\n');
      // index tag3ds from each file
      for (const csvLine of csvLines) {
        if (csvLine) {
          const cols = csvLine.split(';');
          const id = +cols[0];
          const tag3d = cols[1];
          const tagGraph = cols[2];
          const tagObj = tag3dIndex.has(tag3d)
            ? tag3dIndex.get(tag3d)
            : new Map();
          tagObj.set(ID, id);
          tagObj.set(TAG3D, tag3d);
          tagObj.set(TAGGRAPH, tagGraph);
          tagObj.set('category', key);
          tag3dIndex.set(tag3d, tagObj);
        }
      }
      // augment everything with related translations
      for (const locale of LOCALES) {
        const localeFilePath = path.resolve(
          TRANSLATIONS_DIRECTORY,
          `graph_${key}.${locale}.xlf`,
        );
        try {
          const content = await readFile(localeFilePath, FILE_OPTS);
          const localizedRes = parseXliffString(content);
          for (const [tag3d, trans] of localizedRes) {
            const it = tag3dIndex.get(tag3d);
            it.set(locale, trans);
          }
        } catch (err) {}
      }
    }
    // by id index if present in files
    for (const [tag3d, val] of tag3dIndex) {
      if (val.has(ID) && !!val.get(ID)) {
        idIndex.set(val.get(ID), val);
      }
    }
    // by tagGraph index if present in files
    for (const [tag3d, val] of tag3dIndex) {
      if (val.has(TAGGRAPH) && !!val.get(TAGGRAPH)) {
        tagGraphIndex.set(val.get(TAGGRAPH), val);
      }
    }
    initialized = true;
  }
};

const getItemPropFromIndex = async (index, prop, search) => {
  await init();
  return index.has(search) ? index.get(search).get(prop) : undefined;
};

/**
 * Get a 3d tag from a id
 * @param {number} id
 */
const getTag3dFromId = id => getItemPropFromIndex(idIndex, TAG3D, id);

/**
 * Get a tag graph from a id
 * @param {number} id
 */
const getTagGraphFromId = id => getItemPropFromIndex(idIndex, TAGGRAPH, id);

/**
 * Get a id from tag 3d
 * @param {string} tag3d
 */
const getIdFromTag3d = tag3d => getItemPropFromIndex(tag3dIndex, ID, tag3d);

/**
 * Get tag graph from tag 3d
 * @param {string} tag3d
 */
const getTagGraphFromTag3d = tag3d =>
  getItemPropFromIndex(tag3dIndex, TAGGRAPH, tag3d);

/**
 * Get translated tag graph from tag 3d
 * @param {string} tag3d
 * @param {string} locale,
 * @param {Object} options
 */
const getTranslationFromTag3d = async (
  tag3d,
  locale,
  options = { fallbackLocale: undefined },
) => {
  await init();
  let ret;
  const { fallbackLocale } = options;
  if (tag3dIndex.has(tag3d)) {
    const item = tag3dIndex.get(tag3d);
    if (item.has(locale)) {
      ret = item.get(locale);
    }
    if (!ret && fallbackLocale) {
      ret = item.get(fallbackLocale);
    }
  }
  return ret;
};

/**
 * Get a id from a tag graph
 * @param {string} tagGraph
 */
const getIdFromTagGraph = tagGraph =>
  getItemPropFromIndex(tagGraphIndex, ID, tagGraph);

/**
 * Get a 3d tag from a tag graph
 * @param {string} tagGraph
 */
const getTag3dFromTagGraph = tagGraph =>
  getItemPropFromIndex(tagGraphIndex, TAG3D, tagGraph);

/**
 * Get all tags of a category
 * @param {string} category
 */
const getAllFromCategory = async category => {
  await init();
  const ret = new Map();
  for (const [tag3d, item] of tag3dIndex) {
    if (item.get('category') === category) {
      ret.set(tag3d, item);
    }
  }
  return ret;
};

module.exports = {
  CATEGORIES_KEYS,
  CATEGORIES,
  tag3dIndex,
  idIndex,
  tagGraphIndex,
  init,
  getTag3dFromId,
  getTagGraphFromId,
  getIdFromTag3d,
  getTagGraphFromTag3d,
  getTranslationFromTag3d,
  getIdFromTagGraph,
  getTag3dFromTagGraph,
  getAllFromCategory,
};
