<?php

namespace AlloCine\SecureApiBundle\Tests\QueryBuilder;

use AlloCine\SecureApiBundle\Api\Social\ReviewApi;
use AlloCine\SecureApiBundle\QueryBuilder\ReviewQueryBuilder;

class ReviewQueryBuilderTest extends \PHPUnit_Framework_TestCase
{
    public function testNominalCase()
    {
        $mock = $this->getReviewApiMock([
            'filter'    => ReviewApi::FILTER_PRESS,
            'rating'    => 2,
            'webrating' => '1',
            'subject'   => 'movie:1',
            'order'     => 'helpfulasc',
            'page'      => 4,
            'count'     => 10,
            'season'    => null,
            'episode'   => null

        ]);

        $builder = new ReviewQueryBuilder($mock);
        $builder
            ->filter(ReviewApi::FILTER_PRESS)
            ->subject('movie', 1)
            ->range(30, 10)
            ->order('helpful')
            ->onlyRating(2)
            ->execute()
        ;
    }

    public function testNegativeRating()
    {
        $mock = $this->getReviewApiMock([
            'filter'    => null,
            'rating'    => 0.5,
            'webrating' => '1',
            'subject'   => null,
            'order'     => null,
            'page'      => null,
            'count'     => null,
            'season'    => null,
            'episode'   => null
        ]);

        $builder = new ReviewQueryBuilder($mock);
        $builder->onlyRating(-1);
        $builder->execute();
    }

    public function testRatingOverflow()
    {
        $mock = $this->getReviewApiMock([
            'filter'    => null,
            'rating'    => 5,
            'webrating' => '1',
            'subject'   => null,
            'order'     => null,
            'page'      => null,
            'count'     => null,
            'season'    => null,
            'episode'   => null
        ]);

        $builder = new ReviewQueryBuilder($mock);
        $builder->onlyRating(10);
        $builder->execute();
    }

    public function testPageOverflow()
    {
        $mock = $this->getReviewApiMock([
            'filter'    => null,
            'rating'    => null,
            'webrating' => null,
            'subject'   => null,
            'order'     => null,
            'page'      => ReviewQueryBuilder::API_MAX_INT,
            'count'     => 10,
            'season'    => null,
            'episode'   => null
        ]);

        $builder = new ReviewQueryBuilder($mock);
        $builder->page(ReviewQueryBuilder::API_MAX_INT + 1000);
        $builder->execute();
    }

    public function testNegativePage()
    {
        $mock = $this->getReviewApiMock([
            'filter'    => null,
            'rating'    => null,
            'webrating' => null,
            'subject'   => null,
            'order'     => null,
            'page'      => 1,
            'count'     => 10,
            'season'    => null,
            'episode'   => null
        ]);

        $builder = new ReviewQueryBuilder($mock);
        $builder->page(-10);
        $builder->execute();
    }

    public function testCallback()
    {
        $mock = $this->getReviewApiMock();
        $mock
            ->expects($this->once())
            ->method('executeFromBuilder')
            ->willReturn(new \stdClass)
        ;

        $passed = false;

        $builder = new ReviewQueryBuilder($mock);
        $builder->callback(function ($result) use (&$passed) {
            $passed = true;
            $this->assertEquals(new \stdClass, $result);
        });
        $builder->execute();

        $this->assertTrue($passed);
    }

    /**
     * @param null|array $expectedValues
     *
     * @return ReviewApi A mocked version of ReviewApi
     */
    public function getReviewApiMock($expectedValues = null)
    {
        $mock = $this
            ->getMockBuilder('AlloCine\SecureApiBundle\Api\Social\ReviewApi')
            ->disableOriginalConstructor()
            ->getMock()
        ;

        if ($expectedValues) {
            $mock
                ->expects($this->once())
                ->method('executeFromBuilder')
                ->with($expectedValues)
                ->willReturn(new \stdClass)
            ;
        }

        return $mock;
    }
}
