<?php

namespace AlloCine\SecureApiBundle\QueryBuilder;

use AlloCine\SecureApiBundle\Api\Standard\TheaterApi;

class TheaterQueryBuilder
{
    const COUNT = 10;

    /**
     * @var TheaterApi
     */
    private $api;

    /**
     * @var callable
     */
    private $callback;

    /**
     * @var integer
     */
    private $count;

    /**
     * @var string
     */
    private $display;

    /**
     * @var string
     */
    private $filter;

    /**
     * @var \Datetime
     */
    private $from;

    /**
     * @var integer
     */
    private $geocode;

    /**
     * @var float
     */
    private $lat;

    /**
     * @var float
     */
    private $long;

    /**
     * @var string
     */
    private $location;

    /**
     * @var string
     */
    private $membercard;

    /**
     * @var boolean
     */
    private $original;

    /**
     * @var integer
     */
    private $radius;

    /**
     * @var string[]
     */
    private $theaters;

    /**
     * @var integer
     */
    private $zip;

    /**
     * @param TheaterApi $api
     */
    public function __construct(TheaterApi $api)
    {
        $this->api = $api;
        $this->theaters = [];
        $this->count = self::COUNT;
    }

    /**
     * @param string $display
     *
     * @return TheaterQueryBuilder
     */
    public function setDisplay($display)
    {
        $this->display = $display;

        return $this;
    }

    /**
     * @param integer $count
     *
     * @return TheaterQueryBuilder
     */
    public function setCount($count)
    {
        $this->count = $count;

        return $this;
    }

    /**
     * @param string $filter
     *
     * @return TheaterQueryBuilder
     */
    public function setFilter($filter)
    {
        $this->filter = $filter;

        return $this;
    }

    /**
     * @param \DateTime $from
     *
     * @return TheaterQueryBuilder
     */
    public function setFrom(\DateTime $from)
    {
        $this->from = $from;

        return $this;
    }

    /**
     * @param integer $geocode
     *
     * @return TheaterQueryBuilder
     */
    public function setGeocode($geocode)
    {
        $this->geocode = $geocode;

        return $this;
    }

    /**
     * @param string $location
     *
     * @return TheaterQueryBuilder
     */
    public function setLocation($location)
    {
        $this->location = $location;

        return $this;
    }

    /**
     * @param float $lat
     * @param float $long
     *
     * @return TheaterQueryBuilder
     */
    public function setGeo($lat, $long)
    {
        $this->lat  = $lat;
        $this->long = $long;

        return $this;
    }

    /**
     * @param string $membercard
     */
    public function setMembercard($membercard)
    {
        $this->membercard = $membercard;

        return $this;
    }

    /**
     * @param boolean $use
     *
     * @return TheaterQueryBuilder
     */
    public function useOriginalVersion($use = true)
    {
        $this->original = $use;

        return $this;
    }

    /**
     * @param integer $radius
     */
    public function setRadius($radius)
    {
        $this->radius = $radius;

        return $this;
    }

    /**
     * @param string[] $theaters
     */
    public function setTheaters(array $theaters)
    {
        $this->theaters = $theaters;

        return $this;
    }

    /**
     * @param integer $zip
     */
    public function setZip($zip)
    {
        $this->zip = $zip;

        return $this;
    }

    /**
     * If provided, the execute method will call this before
     * returning the api call.
     *
     * @param callable $callback
     *
     * @return TheaterQueryBuilder
     */
    public function callback(callable $callback)
    {
        $this->callback = $callback;

        return $this;
    }

    /**
     * @return mixed|null
     */
    public function execute()
    {
        $theaters = count($this->theaters) ? implode(',', $this->theaters) : null;

        $result = $this->api->executeFromBuilder([
            'count'      => $this->count,
            'display'    => $this->display,
            'filter'     => $this->filter,
            'from'       => $this->from ? $this->from->format('Y-m-d') : null,
            'geocode'    => $this->geocode,
            'lat'        => $this->lat,
            'location'   => $this->location,
            'long'       => $this->long,
            'membercard' => $this->membercard,
            'oversion'   => $this->original ? '1' : null,
            'radius'     => $this->radius,
            'theater'    => $theaters,
            'zip'        => $this->zip
        ]);

        if ($this->callback) {
            $result = call_user_func($this->callback, $result);
        }

        return $result;
    }
}
