<?php

namespace AlloCine\SecureApiBundle\QueryBuilder;

use AlloCine\SecureApiBundle\Api\Standard\ShowtimeApi;

class ShowtimeQueryBuilder
{
    const API_MAX_INT = 2147483646;

    const ORDER_AS_IS = 'asis';
    const ORDER_NAME = 'name';

    const MEMBER_LEPASS = 'lepass';
    const MEMBER_UGC = 'ugcillimite';
    const MEMBER_CINECHEQUE = 'cinecheque';

    const DISPLAY_ANALOG = 'analog';
    const DISPLAY_DIGITAL = 'digital';
    const DISPLAY_3D = '3d';
    const DISPLAY_IMAX = 'imax';
    const DISPLAY_IMAX3D = 'imax3d';

    /**
     * @var ReviewApi
     */
    private $api;

    /**
     * @var callable
     */
    private $callback;

    /**
     * @var \DateTime
     */
    private $date;

    /**
     * @var string
     */
    private $display;

    /**
     * @var integer[]
     */
    private $distributors;

    /**
     * @var float
     */
    private $lat;

    /**
     * @var float
     */
    private $long;

    /**
     * @var string
     */
    private $location;

    /**
     * @var string
     */
    private $memberCard;

    /**
     * @var integer
     */
    private $movie;

    /**
     * @var string
     */
    private $order;

    /**
     * @var integer
     */
    private $radius;

    /**
     * @var string[]
     */
    private $theaters;

    /**
     * @var integer
     */
    private $zip;

    /**
     * @var boolean
     */
    private $original;

    /**
     * @var boolean
     */
    private $now;

    /**
     * @var integer
     */
    private $timerange;

    /**
     * @var integer
     */
    private $triptime;

    /**
     * @var integer
     */
    private $page;

    /**
     * Deliver showtimes before release dates when true
     * not a filter at all, but API call mentions this as a filter 'filter=premiere'
     * @var boolean
     */
    private $premiereFilter = false;

    /**
     * @var integer
     */
    private $count;

    /**
     * @param ShowtimeApi $api
     */
    public function __construct(ShowtimeApi $api)
    {
        $this->api = $api;
        $this->distributors = [];
        $this->theaters = [];
    }

    /**
     * @param \DateTime $date
     *
     * @return ShowtimeQueryBuilder
     */
    public function setDate(\DateTime $date)
    {
        $this->date = $date;

        return $this;
    }

    /**
     * @param string $display
     *
     * @return ShowtimeQueryBuilder
     */
    public function setDisplay($display)
    {
        $this->display = $display;

        return $this;
    }

    /**
     * @param integer[] $display
     *
     * @return ShowtimeQueryBuilder
     */
    public function setDistributors(array $distributors)
    {
        $this->distributors = $distributors;

        return $this;
    }

    /**
     * @param float $lat
     * @param float $long
     *
     * @return ShowtimeQueryBuilder
     */
    public function setGeo($lat, $long)
    {
        $this->lat = $lat;
        $this->long = $long;

        return $this;
    }

    /**
     * @param string $location
     *
     * @return ShowtimeQueryBuilder
     */
    public function setLocation($location)
    {
        $this->location = $location;

        return $this;
    }

    /**
     * @param string $memberCard
     *
     * @return ShowtimeQueryBuilder
     */
    public function setMembercard($memberCard)
    {
        $this->memberCard = $memberCard;

        return $this;
    }

    /**
     * @param integer $movie
     *
     * @return ShowtimeQueryBuilder
     */
    public function setMovie($movie)
    {
        $this->movie = $movie;

        return $this;
    }

    /**
     * @param string $order
     *
     * @return ShowtimeQueryBuilder
     */
    public function setOrder($order)
    {
        $this->order = $order;

        return $this;
    }

    /**
     * @param integer $radius
     *
     * @return ShowtimeQueryBuilder
     */
    public function setRadius($radius)
    {
        $this->radius = $radius;

        return $this;
    }

    /**
     * @param string[] $theaters
     *
     * @return ShowtimeQueryBuilder
     */
    public function setTheaters(array $theaters)
    {
        $this->theaters = $theaters;

        return $this;
    }

    /**
     * @param integer $zip
     *
     * @return ShowtimeQueryBuilder
     */
    public function setZip($zip)
    {
        $this->zip = $zip;

        return $this;
    }

    /**
     * @param boolean $use
     *
     * @return ShowtimeQueryBuilder
     */
    public function useOriginalVersion($use = true)
    {
        $this->original = $use;

        return $this;
    }

    /**
     * @param integer $timerange
     *
     * @return ShowtimeQueryBuilder
     */
    public function setTimerange($timerange)
    {
        $this->timerange = $timerange;

        return $this;
    }

    /**
     * @param integer $timerange
     *
     * @return ShowtimeQueryBuilder
     */
    public function setTriptime($triptime)
    {
        $this->triptime = $triptime;

        return $this;
    }

    /**
     * @param integer $page
     * @param integer $count
     *
     * @return ShowtimeQueryBuilder
     */
    public function page($page, $count = 10)
    {
        // Force pagination to be between [1, 2^32[
        $page = max(min($page, self::API_MAX_INT), 1);

        $this->page = $page;
        $this->count = $count;

        return $this;
    }

    /**
     * If provided, the execute method will call this before
     * returning the api call.
     *
     * @param callable $callback
     *
     * @return ReviewQueryBuilder
     */
    public function callback(callable $callback)
    {
        $this->callback = $callback;

        return $this;
    }

    /**
     * Set filter API param to 'premiere' for getting the first dates before release date
     *
     * @param boolean $premiere
     *
     * @return self
     */
    public function setPremiereFilter(bool $premiereFilter)
    {
        $this->premiereFilter = $premiereFilter;

        return $this;
    }

    /**
     * @return mixed|null
     */
    public function execute()
    {
        $distributors = count($this->distributors) ? implode(',', $this->distributors) : null;
        $theaters = count($this->theaters) ? implode(',', $this->theaters) : null;
        $params = [
            'date' => $this->date ? $this->date->format('Y-m-d') : null,
            'display' => $this->display,
            'distributors' => $distributors,
            'lat' => $this->lat,
            'long' => $this->long,
            'location' => $this->location,
            'membercard' => $this->memberCard,
            'movie' => $this->movie,
            'order' => $this->order,
            'radius' => $this->radius,
            'theaters' => $theaters,
            'zip' => $this->zip,
            'oversion' => $this->original ? '1' : null,
            'triptime' => $this->triptime,
            'timerange' => $this->timerange,
            'page' => $this->page,
            'count' => $this->count
        ];

        if ($this->premiereFilter) {
            $params['filter'] = 'premiere';
        }

        $result = $this->api->executeFromBuilder($params);

        if ($this->callback) {
            $result = call_user_func($this->callback, $result);
        }

        return $result;
    }
}
