<?php

namespace AlloCine\SecureApiBundle\Client;

use GuzzleHttp\Client;
use GuzzleHttp\HandlerStack;
use Csa\GuzzleHttp\Middleware\History\History;

/**
 * Class MultiRegionApiClient
 */
class MultiRegionApiClient
{
    /**
     * @param array
     */
    private $regions;

    /**
     * @var string;
     */
    private $defaultRegion;

    /**
     * @var string;
     */
    private $token;

    /**
     * @var array
     */
    private $clients = [];

    /**
     * @var HandlerStack
     */
    private $handler;

    /**
     * @var History
     */
    private $collector;

    /**
     * @param HandlerStack|array $guzzleConfig
     */
    public function __construct($guzzleConfig = null)
    {
        if (is_array($guzzleConfig)) {
            $guzzleConfig = array_key_exists('handler', $guzzleConfig) ? $guzzleConfig['handler'] : null;
        }

        $this->handler = $guzzleConfig;
    }

    /**
     * @return HandlerStack
     */
    public function getConfig()
    {
        return $this->handler;
    }

    /**
     * @param array $regions
     *
     * @return MultiRegionApiClient
     */
    public function setRegions(array $regions)
    {
        $this->regions = $regions;

        return $this;
    }

    /**
     * @param string $region
     *
     * @throws InvalidArgumentException If the region is not found.
     *
     * @return MultiRegionApiClient
     */
    public function setDefaultRegion($region)
    {
        if (! array_key_exists($region, $this->regions)) {
            throw new \InvalidArgumentException(sprintf('The region "%s" does not exists.', $region));
        }

        $this->defaultRegion = $region;

        return $this;
    }

    /**
     * @param History $collector
     */
    public function setCollector(History $collector)
    {
        $this->collector = $collector;
    }

    /**
     * Get a regional client, create it if needed
     *
     * @param string $region
     *
     * @throws InvalidArgumentException If the region is not found.
     *
     * @return Client
     */
    public function getClient($region = null)
    {
        $region = $region ? $region : $this->defaultRegion;

        if (!array_key_exists($region, $this->regions)) {
            throw new \InvalidArgumentException(sprintf('The region "%s" does not exists.', $region));
        }

        if (!array_key_exists($region, $this->clients)) {
            $options = [
                'handler'  => $this->handler,
                'base_uri' => $this->regions[$region]['domain'],
                'timeout'  => 2.0,
            ];

            if (!is_null($this->collector)) {
                $options['on_stats'] = [$this->collector, 'addStats'];
            }

            $this->clients[$region] = new Client($options);
        }

        return $this->clients[$region];
    }

    /**
     * Set or override a client for a region. Usefull for injecting mocks in unit tests.
     *
     * @param stirng $region
     * @param Client $client
     *
     * @return MultiRegionApiClient
     */
    public function setClient($region, Client $client)
    {
        $this->clients[$region] = $client;

        return $this;
    }

    /**
     * @param string $path
     * @param array  $options
     *
     * @return \stdClass
     */
    public function get($path, array $options = [])
    {
        $region = array_key_exists('region', $options) ? $options['region'] : $this->defaultRegion;
        $client = $this->getClient($region);

        if (! array_key_exists('query', $options)) {
            $options['query'] = [];
        }

        if (array_key_exists('partner', $this->regions[$region])) {
            $options['query']['partner'] = $this->regions[$region]['partner'];
        }

        $options['query']['format'] = 'json';

        if (!is_null($this->token)) {
            $options['cookies'] = new CookieJar(
                true,
                [
                    new SetCookie([
                        'Name'      => 'ACCT',
                        'Value'     => $this->token,
                        'Domain'    =>
                            implode(
                                '.',
                                array_slice(
                                    explode('.', $this->regions[$region]['domain']),
                                    -2
                                )
                            ),
                    ])
                ]
            );
        }

        return $client->get($path, $options);
    }

    /**
     * @param string $token
     *
     * @return MultiRegionApiClient
     */
    public function setToken($token)
    {
        $this->token = $token;

        return $this;
    }
}
