<?php

namespace AlloCine\SecureApiBundle\Tests\Client;

use AlloCine\SecureApiBundle\Client\MultiRegionApiClient;
use GuzzleHttp\Client;
use GuzzleHttp\HandlerStack;
use GuzzleHttp\Handler\MockHandler;
use GuzzleHttp\Psr7\Response;
use Guzzle\Plugin\Mock\MockPlugin;

class MultiRegionApiClientTest extends \PHPUnit_Framework_TestCase
{
    public function testGetWithDefaultRegion()
    {
        $client = $this->getClient('the response');
        $client->setDefaultRegion('fr');

        $response = (string)$client->get('/foo')->getBody();

        $this->assertSame('fr : the response', $response);
    }

    public function testGetWithOverridedRegion()
    {
        $client = $this->getClient('the response');
        $client->setDefaultRegion('fr');

        $response = (string)$client->get('/foo', ['region' => 'es'])->getBody();

        $this->assertSame('es : the response', $response);
    }

    /**
     * @expectedException \InvalidArgumentException
     * @expectedExceptionMessage The region "en" does not exists.
     */
    public function testGetWithInvalidRegion()
    {
        $client = $this->getClient('the response');
        $client->setDefaultRegion('fr');

        $client->get('/foo', ['region' => 'en']);
    }

    /**
     * @expectedException \InvalidArgumentException
     * @expectedExceptionMessage The region "en" does not exists.
     */
    public function testInvalidDefaultRegion()
    {
        $client = $this->getClient();
        $client->setDefaultRegion('en');
    }

    /**
     * @expectedException \InvalidArgumentException
     * @expectedExceptionMessage The region "en" does not exists.
     */
    public function testInvalidClient()
    {
        $client = $this->getClient();
        $client->getClient('en');
    }

    /**
     * @param string  $result
     * @param integer $statusCode
     *
     * @return MultiRegionApiClient
     */
    private function getClient($result = '', $statusCode = 200)
    {
        $client = new MultiRegionApiClient();

        $client->setRegions([
            'fr' => [
                'domain'  => 'http://api.allocine.fr/rest/v3/',
                'partner' => '000042784970'
            ],
            'es' => [
                'domain'  => 'http://api.sensacine.com/rest/v3/',
                'partner' => '000042784970'
            ]
        ]);

        $client
            ->setClient('fr', new Client([
                'handler' => new MockHandler([
                    new Response($statusCode, [], sprintf('fr : %s', $result)),
                ])
            ]))
            ->setClient('es', new Client([
                'handler' => new MockHandler([
                    new Response($statusCode, [], sprintf('es : %s', $result)),
                ])
            ]))
        ;

        return $client;
    }
}
