<?php

namespace AlloCine\SecureApiBundle\QueryBuilder;

use AlloCine\SecureApiBundle\Api\Social\ReviewApi;

class ReviewQueryBuilder
{
    const API_MAX_INT = 2147483646;

    /**
     * @var string
     */
    private $filter;

    /**
     * @var string
     */
    private $subject;

    /**
     * @var string
     */
    private $order;

    /**
     * @var integer
     */
    private $page;

    /**
     * @var integer
     */
    private $count;

    /**
     * @var integer
     */
    private $rating;

    /**
     * @var integer
     */
    private $season;

    /**
     * @var integer
     */
    private $episode;

    /**
     * @var boolean
     */
    private $ratingFilterMode;

    /**
     * @var ReviewApi
     */
    private $api;

    /**
     * @var callable
     */
    private $callback;

    /**
     * @param ReviewApi $api
     */
    public function __construct(ReviewApi $api)
    {
        $this->api = $api;
    }

    /**
     * @param string $filter
     */
    public function filter($filter)
    {
        $this->filter = $filter;

        return $this;
    }

    /**
     * @param string  $type
     * @param integer $id
     *
     * @return ReviewQueryBuilder
     */
    public function subject($type, $id)
    {
        $this->subject = sprintf('%s:%s', $type, $id);

        return $this;
    }

    /**
     * @param string  $offset
     * @param integer $limit
     *
     * @return ReviewQueryBuilder
     */
    public function range($offset, $limit = 10)
    {
        $this->page = (intval($offset / $limit) + 1);
        $this->count = $limit;

        return $this;
    }

    /**
     * @param integer $page
     * @param integer $count
     *
     * @return ReviewQueryBuilder
     */
    public function page($page, $count = 10)
    {
        // Force pagination to be between [1, 2^32[
        $page = max(min($page, self::API_MAX_INT), 1);

        $this->page = $page;
        $this->count = $count;

        return $this;
    }

    /**
     * @param string $type
     * @param string $direction
     *
     * @return ReviewQueryBuilder
     */
    public function order($type, $direction = 'asc')
    {
        $this->order = sprintf('%s%s', $type, $direction);

        return $this;
    }

    /**
     * @param integer $rating
     *
     * @return ReviewQueryBuilder
     */
    public function onlyRating($rating, $ratingFilterMode = ReviewApi::RATING_FILTER_WEB)
    {
        $this->rating = intval($rating);
        $this->ratingFilterMode = $ratingFilterMode;

        if ($this->rating <= 0) {
            $this->rating = 0.5;
        } elseif ($this->rating > 5) {
            $this->rating = 5;
        }

        return $this;
    }

    /**
     * @param integer $season
     *
     * @return $this
     */
    public function onlySeason($season)
    {
        $this->season = $season;

        return $this;
    }

    /**
     * @param integer $episode
     *
     * @return $this
     */
    public function onlyEpisode($episode)
    {
        $this->episode = $episode;

        return $this;
    }

    /**
     * If provided, the execute method will call this before
     * returning the api call.
     *
     * @param  callable $callback
     *
     * @return ReviewQueryBuilder
     */
    public function callback(callable $callback)
    {
        $this->callback = $callback;

        return $this;
    }

    /**
     * @return mixed|null
     */
    public function execute()
    {
        $result = $this->api->executeFromBuilder([
            'filter' => $this->filter,
            'rating' => $this->rating,
            'season' => $this->season,
            'episode' => $this->episode,
            'webrating' => $this->ratingFilterMode === ReviewApi::RATING_FILTER_WEB ? '1' : null,
            'subject' => $this->subject,
            'order' => $this->order,
            'page' => $this->page,
            'count' => $this->count
        ]);

        if ($this->callback) {
            $result = call_user_func($this->callback, $result);
        }

        return $result;
    }
}
