<?php

namespace Test;

use Allocine\Routing\UrlGenerator;
use Allocine\Routing\UrlMatcher;
use Symfony\Component\Routing\Matcher\Dumper\PhpMatcherDumper;
use Symfony\Component\Routing\RequestContext;
use Symfony\Component\Routing\Route;
use Symfony\Component\Routing\RouteCollection;

class ParameterMappingTest extends \PHPUnit_Framework_TestCase
{
    private $routes;
    private $matcher;

    protected function setUp()
    {
        $this->routes = new RouteCollection();
        $this->routes->add('movies', new Route('/movies/{order}', [ '_mapping' => [
            'order' => [ 'order_rate' => 'ratings', 'order_alpha' => 'alpha' ]
        ] ], [], ['compiler_class' => 'Allocine\Routing\AllocineRouteCompiler']));

        $this->routes->add('movies_fragment', new Route('/listing[/order-{order}]', [ '_mapping' => [
            'order' => [ 'order_rate' => 'ratings', 'order_alpha' => '' ]
        ] ], [], ['compiler_class' => 'Allocine\Routing\AllocineRouteCompiler']));

        $this->matcher = new UrlMatcher($this->routes, new RequestContext(''));
        $this->generator = new UrlGenerator($this->routes, new RequestContext(''));
    }

    public function getMatchingData()
    {
        return [
            [ '/movies/ratings', [
                '_mapping' => [ 'order' => [ 'order_rate' => 'ratings', 'order_alpha' => 'alpha' ]],
                'order' => 'order_rate',
                '_route' => 'movies'
            ]],
            [ '/movies/alpha', [
                '_mapping' => [ 'order' => [ 'order_rate' => 'ratings', 'order_alpha' => 'alpha' ]],
                'order' => 'order_alpha',
                '_route' => 'movies'
            ]],
            [ '/listing', [
                '_mapping' => [ 'order' => [ 'order_rate' => 'ratings', 'order_alpha' => '' ]],
                '_route' => 'movies_fragment',
                'order' => 'order_alpha',
            ]],
            [ '/listing/order-ratings', [
                '_mapping' => [ 'order' => [ 'order_rate' => 'ratings', 'order_alpha' => '' ]],
                'order' => 'order_rate',
                '_route' => 'movies_fragment',
            ]]
        ];
    }

    public function getGenerationData()
    {
        return [
            [ 'movies', [ 'order' => 'order_rate' ], '/movies/ratings' ],
            [ 'movies', [ 'order' => 'order_alpha' ], '/movies/alpha' ],
            [ 'movies_fragment', [ ], '/listing' ],
            [ 'movies_fragment', [ 'order' => 'order_rate' ], '/listing/order-ratings' ],
            [ 'movies_fragment', [ 'order' => 'order_alpha' ], '/listing' ],
        ];
    }

    /**
     * @expectedException Symfony\Component\Routing\Exception\ResourceNotFoundException
     * @expectedExceptionMessage No routes found for "/movies/notallowed".
     */
    public function testNotAllowedParameterMatching()
    {
        $this->matcher->match('/movies/notallowed');
    }

    /**
     * @expectedException Symfony\Component\Routing\Exception\InvalidParameterException
     * @expectedExceptionMessage Parameter "order" is mapped and should be in order_rate, order_alpha ("notallowed" given).
     */
    public function testNotAllowedParameterGeneration()
    {
        $this->generator->generate('movies', [ 'order' => 'notallowed' ]);
    }

    public function testForcedParameterGeneration()
    {
        $this->generator->setStrictRequirements(null);
        $this->assertSame('/movies/forced', $this->generator->generate('movies', [ 'order' => 'forced' ]));
    }

    /**
     * @dataProvider getMatchingData
     */
    public function testMappingMatching($url, $expectedResult)
    {
        $this->assertSame($expectedResult, $this->matcher->match($url));
    }

    /**
     * @dataProvider getGenerationData
     */
    public function testGeneration($route, array $params, $expectedResult)
    {
        $this->assertSame($expectedResult, $this->generator->generate($route, $params));
    }

    protected function tearDown()
    {
        $this->routes = null;
        $this->matcher = null;
    }
}
