<?php

namespace Test;

use Allocine\Routing\UrlGenerator;
use Allocine\Routing\UrlMatcher;
use Symfony\Component\Routing\Matcher\Dumper\PhpMatcherDumper;
use Symfony\Component\Routing\RequestContext;
use Symfony\Component\Routing\Route;
use Symfony\Component\Routing\RouteCollection;

class FragmentRoutingTest extends \PHPUnit_Framework_TestCase
{
    private $routes;
    private $matcher;

    protected function setUp()
    {
        $this->routes = new RouteCollection();
        $this->routes->add('simple', new Route('/foo', []));
        $this->routes->add('simple_arg', new Route('/bar/{arg}', []));
        $this->routes->add('optional', new Route('/baz/{arg}/{arg2}', ['arg' => '1', 'arg2' => '2']));
        $this->routes->add('filtered', new Route('/filters[/arg-{arg}][/arg2-{arg2}]', ['arg' => '1', 'arg2' => '2'], [], ['compiler_class' => 'Allocine\Routing\AllocineRouteCompiler']));
        $this->routes->add('filtered_suffix', new Route('/filters[/arg-{arg}][/arg2-{arg2}]/suffix', ['arg' => '1', 'arg2' => '2'], [], ['compiler_class' => 'Allocine\Routing\AllocineRouteCompiler']));
        $this->routes->add('multi_filters', new Route('/filters[/arg-{arg}]/sep[/arg2-{arg2}]/suffix', ['arg' => '1', 'arg2' => '2'], [], ['compiler_class' => 'Allocine\Routing\AllocineRouteCompiler']));
        $this->routes->add('movies', new Route('/films[/{order}][/genre-{genre}]', ['order' => 'best'], [ 'genre' => '[0-9]+' ], ['compiler_class' => 'Allocine\Routing\AllocineRouteCompiler']));
        $this->routes->add('adoro', new Route('/filmes/todos-filmes/notas-espectadores/[genero-{genre}/][pais-{country}/][decada-{decade}/][ano-{year}/][{type}/]', [], [], ['compiler_class' => 'Allocine\Routing\AllocineRouteCompiler']));
        $this->routes->add('secured', new Route('/login', [], [], [], '', ['https']));
        $this->routes->add('recover', new Route('/recover/token={token}', [], [], [], '', ['https']));
        $this->matcher = new UrlMatcher($this->routes, new RequestContext(''));
        $this->generator = new UrlGenerator($this->routes, new RequestContext(''));
    }

    public function getGenerationData()
    {
        return [
            // Zero value
            ['simple_arg', ['arg' => 0], '/bar/0'],

            // No filters provided.
            ['filtered', [], '/filters'],

            // Filters values equals defaults.
            ['filtered', ['arg' => '1'], '/filters/arg-1'],
            ['filtered', ['arg2' => '2'], '/filters/arg2-2'],
            ['filtered', ['arg' => '1', 'arg2' => '2'], '/filters/arg-1/arg2-2'],

            // Different values than defaults
            ['filtered', ['arg' => '3'], '/filters/arg-3'],
            ['filtered', ['arg2' => '4'], '/filters/arg2-4'],
            ['filtered', ['arg' => '3', 'arg2' => '4'], '/filters/arg-3/arg2-4'],

            // 0 value generation
            ['filtered', ['arg' => 0], '/filters/arg-0'],

            // Suffix
            ['filtered_suffix', [], '/filters/suffix'],
            ['filtered_suffix', ['arg' => '3'], '/filters/arg-3/suffix'],

            // Requirements
            ['movies', [], '/films'],
            ['movies', [ 'genre' => 1 ], '/films/genre-1'],
        ];
    }

    public function getMatchingData()
    {
        return [
            // Simple case
            ['/foo', ['_route' => 'simple']],
            ['/bar/1', ['arg' => '1', '_route' => 'simple_arg']],
            ['/bar/0', ['arg' => '0', '_route' => 'simple_arg']],

            // Symfony's base optional suffix
            ['/baz', [ 'arg' => '1', 'arg2' => '2' , '_route' => 'optional']],
            ['/baz/3', [ 'arg' => '3', 'arg2' => '2' , '_route' => 'optional']],

            // Filtered route
            ['/filters', [ 'arg' => '1', 'arg2' => '2', '_route' => 'filtered']],
            ['/filters/arg-4/arg2-4', [ 'arg' => '4', 'arg2' => '4', '_route' => 'filtered']],
            ['/filters/arg2-4', [ 'arg' => '1', 'arg2' => '4', '_route' => 'filtered']],

            // Filtered route with suffix
            ['/filters/suffix', ['arg' => '1', 'arg2' => '2', '_route' => 'filtered_suffix']],
            ['/filters/arg-4/arg2-4/suffix', ['arg' => '4', 'arg2' => '4', '_route' => 'filtered_suffix']],
            ['/filters/arg2-4/suffix', ['arg' => '1', 'arg2' => '4', '_route' => 'filtered_suffix']],

            // Filtered route with separator and suffix
            ['/filters/sep/suffix', ['arg' => '1', 'arg2' => '2', '_route' => 'multi_filters']],
            ['/filters/arg-4/sep/arg2-4/suffix', ['arg' => '4', 'arg2' => '4', '_route' => 'multi_filters']],
            ['/filters/sep/arg2-4/suffix', ['arg' => '1', 'arg2' => '4', '_route' => 'multi_filters']],
        ];
    }

    public function getRedirectingData()
    {
        return [
            ['/login', ['_route' => 'secured']],
            ['/recover/token=123265451', ['token' => '123265451', '_route' => 'recover']],
        ];
    }
    /**
     * @expectedException Symfony\Component\Routing\Exception\InvalidParameterException
     * @expectedExceptionMessage Parameter "genre" for route "movies" must match "[0-9]+" ("foo" given) to generate a corresponding URL.
     */
    public function testInvalidGenerationRequirements()
    {
        $this->generator->setStrictRequirements(false);
        $this->assertSame('/films/genre-bar', $this->generator->generate('movies', ['genre' => 'bar']));
        $this->generator->setStrictRequirements(true);

        $this->generator->generate('movies', ['genre' => 'foo']);
    }

    /**
     * @dataProvider getGenerationData
     */
    public function testGeneration($route, array $params, $expectedResult)
    {
        $this->assertSame($expectedResult, $this->generator->generate($route, $params));
    }

    /**
     * @dataProvider getMatchingData
     */
    public function testMatching($url, $expectedResult)
    {
        $this->assertSame($expectedResult, $this->matcher->match($url));
    }

    /**
     * @dataProvider getRedirectingData
     */
    public function testRedirect($url, $expectedResult)
    {
        $actual = $this->matcher->match($url);
        $this->assertSame($expectedResult['_route'], $actual['_route']);
        $this->assertStringStartsWith('https', $actual['url']);
    }

    protected function tearDown()
    {
        $this->routes = null;
        $this->matcher = null;
    }
}
