<?php

namespace App\Service;

use AMQPChannel;
use AMQPChannelException;
use AMQPConnection;
use AMQPConnectionException;
use AMQPExchange;
use AMQPExchangeException;
use AMQPQueue;
use AMQPQueueException;
use App\Common\NotifierInterface;
use Elasticsearch\ClientBuilder;
use Exception;
use RuntimeException;
use Swarrot\SwarrotBundle\Broker\UrlParserTrait;

class InitMq
{
    use UrlParserTrait;

    /**
     * @param array $config
     *
     * @throws AMQPConnectionException
     */
    public function launch(array $config, NotifierInterface $referer): void
    {
        $connection = $this->getMqConnection($config['mq']['url']);
        $channel = $this->getMqChannel($connection);

        try {
            foreach ($config['mq']['queues'] as $queue) {
                $referer->notify('<info>Creating ' . $queue['queue_name'] . '</info>');
                $this->createMqExchange($channel, $queue['exchange'], $queue['exchange_type']);
                $this->createMqQueue(
                    $channel,
                    $queue['queue_name'],
                    $queue['exchange'],
                    $queue['routing_key'],
                    $queue['options'] ?? null
                );
            }
        } catch (Exception $exception) {
            throw new RuntimeException(sprintf(
                'Failed to create the exchange and/or the queue because of %s exception. %s',
                get_class($exception),
                $exception->getMessage()
            ));
        }

        $referer->notify('<info>Exchanges and queues created successfuly</info>');
    }

    /**
     * @param string $url
     *
     * @return AMQPConnection
     *
     * @throws AMQPConnectionException
     */
    private function getMqConnection(string $url): AMQPConnection
    {
        $params = $this->parseUrl($url);

        $connection = new AMQPConnection($params);
        $connection->connect();

        return $connection;
    }

    /**
     * @param AMQPConnection $connection
     *
     * @throws AMQPConnectionException
     *
     * @return AMQPChannel
     */
    private function getMqChannel(AMQPConnection $connection): AMQPChannel
    {
        $channel = new AMQPChannel($connection);
        $channel->setPrefetchCount(1);

        return $channel;
    }

    /**
     * @param AMQPChannel $channel
     * @param string $exchangeName
     * @param string $exchangeType
     *
     * @throws AMQPChannelException
     * @throws AMQPConnectionException
     * @throws AMQPExchangeException
     */
    private function createMqExchange(AMQPChannel $channel, string $exchangeName, string $exchangeType): void
    {
        $exchange = new AMQPExchange($channel);
        $exchange->setName($exchangeName);
        $exchange->setType(constant($exchangeType));
        $exchange->setFlags(AMQP_DURABLE);
        $exchange->declareExchange();
    }

    /**
     * @param AMQPChannel $channel
     * @param string $queueName
     * @param string $exchangeName
     * @param string $routingKey
     * @param null $options
     *
     * @throws AMQPChannelException
     * @throws AMQPConnectionException
     * @throws AMQPQueueException
     */
    private function createMqQueue(
        AMQPChannel $channel,
        string $queueName,
        string $exchangeName,
        string $routingKey,
        $options = null
    ): void
    {
        if ((null !== $options) && (is_string($options))) {
            $options = json_decode($options, true);
        }

        $q = new AMQPQueue($channel);
        $q->setName($queueName);
        $q->setFlags(AMQP_DURABLE);
        if (null !== $options) {
            $q->setArguments($options);
        }
        $q->declareQueue();
        $q->bind($exchangeName, $routingKey);
    }
}
