<?php

namespace App\Service;

use App\Common\Constant;
use Exception;
use PommProject\Foundation\Pomm;
use Psr\Log\LoggerInterface;
use Symfony\Component\Process\Exception\ProcessFailedException;
use Symfony\Component\Process\Process;

class Slot
{
    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var string
     */
    private $databaseDsn;

    /**
     * @var string
     */
    private $slotName;

    /**
     * @var string
     */
    private $slotTables;

    /**
     * @var array
     */
    private $pgRecvLogicalParameters;

    /**
     * @param LoggerInterface $logger
     * @param string $databaseDsn
     * @param string $slotName
     * @param string $slotTables
     */
    public function __construct(
        LoggerInterface $logger,
        Pomm $pomm,
        string $databaseDsn,
        string $slotName,
        string $slotTables
    ) {
        $this->logger = $logger;
        $this->databaseDsn = $databaseDsn;
        $this->slotName = $slotName;
        $this->slotTables = $slotTables;
        $this->pommSession = $pomm->getSession('master');
    }

    /***
     * @return Process
     *
     * @throws Exception
     */
    public function createSlot(): Process
    {
        $this->getRecvlogicalParameters();

        // In case it exists, we ignore the error
        $createSlotProcess = new Process([
            'pg_recvlogical',
            '-h',
            $this->pgRecvLogicalParameters['host'],
            '-d',
            $this->pgRecvLogicalParameters['dbname'],
            '-p',
            $this->pgRecvLogicalParameters['port'],
            '--slot',
            $this->slotName,
            '--create-slot',
            '-P',
            'wal2json',
            '-U',
            $this->pgRecvLogicalParameters['user'],
            '--if-not-exists',
        ]);
        $createSlotProcess->setTimeout(600);
        $createSlotProcess->setIdleTimeout(600);
        if (!empty($this->pgRecvLogicalParameters['password'])) {
            $createSlotProcess->setEnv(['PGPASSWORD' => $this->pgRecvLogicalParameters['password']]);
        }
        $this->logger->notice('Create slot if not exists', [
            'component' => Constant::MQ_COMPONENT_NAME,
            'slot' => $this->slotName,
        ]);

        $createSlotProcess->run();
        if ('' !== $createSlotProcess->getOutput()) {
            $this->logger->notice($createSlotProcess->getOutput(), [
                'component' => selConstantf::MQ_COMPONENT_NAME,
                'slot' => $this->slotName,
            ]);
        }

        return $createSlotProcess;
    }

    /**
     * @return Process
     *
     * @throws Exception
     */
    public function getListenSlotProcess(): Process
    {
        $params = [
            'pg_recvlogical',
            '-h',
            $this->getRecvlogicalParameters()['host'],
            '-d',
            $this->getRecvlogicalParameters()['dbname'],
            '-p',
            $this->getRecvlogicalParameters()['port'],
            '--slot',
            $this->slotName,
            '-U',
            $this->getRecvlogicalParameters()['user'],
            '--start',
            '-o',
            sprintf('add-tables=%s', $this->slotTables),
            '-o',
            'include-types=0',
            '-o',
            'include-timestamp=true',
        ];
        $params[] = '-f';
        $params[] = '-';
        $listenSlotProcess = new Process($params);
        if (!empty($this->getRecvlogicalParameters()['password'])) {
            $listenSlotProcess->setEnv(['PGPASSWORD' => $this->getRecvlogicalParameters()['password']]);
        }
        // disable timeout
        $listenSlotProcess->setTimeout(null);
        $listenSlotProcess->setIdleTimeout(null);

        return $listenSlotProcess;
    }

    /**
     * @throws Exception
     *
     * @return int
     */
    public function cleanSlot(): int
    {
        $this->logger->notice('Cleaning slot', [
            'component' => Constant::MQ_COMPONENT_NAME,
            'slot'      => $this->slotName
        ]);

        $this->pommSession->getConnection()->executeAnonymousQuery(
            sprintf(<<<EOF
SELECT pg_terminate_backend(pid)  
FROM pg_stat_activity pa 
INNER JOIN pg_replication_slots pr ON pa.pid=pr.active_pid 
WHERE pr.slot_name='%s'
EOF
,
                $this->slotName
            )
        );

        // Dropping the slot
        $dropSlotProcess = new Process([
            'pg_recvlogical',
            '-h',
            $this->getRecvlogicalParameters()['host'],
            '-d',
            $this->getRecvlogicalParameters()['dbname'],
            '-p',
            $this->getRecvlogicalParameters()['port'],
            '--slot',
            $this->slotName,
            '--drop-slot',
            '-U',
            $this->getRecvlogicalParameters()['user'],
        ]);

        if (!empty($this->getRecvlogicalParameters()['password'])) {
            $dropSlotProcess->setEnv(['PGPASSWORD' => $this->getRecvlogicalParameters()['password']]);
        }
        $dropSlotProcess->run();

        if (!$dropSlotProcess->isSuccessful()) {
            throw new ProcessFailedException($dropSlotProcess);
        }

        $this->logger->notice('Cleaned slot', [
            'component' => Constant::MQ_COMPONENT_NAME,
            'slot'      => $this->slotName
        ]);

        return 0;
    }

    /**
     * @throws Exception
     */
    private function getRecvlogicalParameters(): array
    {
        if (null === $this->pgRecvLogicalParameters) {
            $regex = "/pgsql:\/\/(?P<user>[^:@]+)(:(?P<password>[^:@]+)?)?"
                . "@(?P<host>[a-z0-9.-]+):(?P<port>[0-9]+)\/(?P<dbname>[a-z0-9._-]+)/";
            if (preg_match($regex, $this->databaseDsn, $matches) == false) {
                throw new Exception('DatabaseDsn does not match expected format');
            }

            $this->pgRecvLogicalParameters = [
                'host' => $matches['host'],
                'slot' => $this->slotName,
                'dbname' => $matches['dbname'],
                'user' => $matches['user'],
                'password' => $matches['password'],
                'port' => $matches['port'],
            ];
        }

        return $this->pgRecvLogicalParameters;
    }

    /**
     * @return string
     */
    public function getSlotName(): string
    {
        return $this->slotName;
    }
}
