<?php

namespace App\Service\Publisher;

use App\Common\Constant;
use App\Swarrot\DbzCustomProviderFactory;
use PhpAmqpLib\Message\AMQPMessage;
use Psr\Log\LoggerInterface;
use Swarrot\Broker\Message;
use Swarrot\SwarrotBundle\Broker\Publisher;

class MQPublisher implements PublisherInterface
{
    private const MESSAGE_TYPE = 'event_producer';

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var Publisher
     */
    private $mqPublisher;

    /**
     * @var string
     */
    private $routingKeyPrefix;

    /**
     * @var DbzCustomProviderFactory
     */
    private $brokerFactory;

    public function __construct(
        Publisher $mqPublisher,
        LoggerInterface $logger,
        DbzCustomProviderFactory $brokerFactory,
        string $routingKeyPrefix
    ) {
        $this->logger = $logger;
        $this->mqPublisher = $mqPublisher;
        $this->routingKeyPrefix = $routingKeyPrefix;
        $this->brokerFactory = $brokerFactory;
    }

    public function publish(\stdClass $walChange): void
    {
        $jsonWalChange = json_encode($walChange);
        $body = $this->createMessageFromWalChange($walChange);
        $hash = sha1($jsonWalChange);

        $message = new Message($body, [
            'headers' => [
                'hash' => $hash,
            ],
            'delivery_mode' => AMQPMessage::DELIVERY_MODE_PERSISTENT
        ]);

        try {
            $this->publishMessage($message, $walChange);
        } catch (\Exception $publishException) {
            $this->logger->warning($publishException->getMessage(), [
                'component' => Constant::MQ_COMPONENT_NAME,
                'exception_class' => get_class($publishException),
            ]);

            $this->recoverConnection();

            try {
                $this->publishMessage($message, $walChange, true);
            } catch (\Exception $republishException) {
                $this->logger->critical($republishException->getMessage(), [
                    'component' => Constant::MQ_COMPONENT_NAME,
                    'exception_class' => get_class($republishException),
                ]);
            }
        }
    }

    private function createMessageFromWalChange(\stdClass $walChange): string
    {
        $ret = new \stdClass();
        $ret->kind = $walChange->kind;
        $ret->schema = $walChange->schema;
        $ret->table = $walChange->table;
        $ret->wal_timestamp = (new \DateTime($walChange->timestamp))->format('c');
        $ret->parse_timestamp = (new \DateTime())->format('c');

        if ($walChange->kind === 'insert' || $walChange->kind === 'update') {
            $ret->new = new \stdClass();

            foreach ($walChange->columnnames as $id => $name) {
                $ret->new->{$name} = $walChange->columnvalues[$id];
            }
        }

        if ($walChange->kind === 'delete' || $walChange->kind === 'update') {
            $ret->old = new \stdClass();

            foreach ($walChange->oldkeys->keynames as $id => $name) {
                $ret->old->{$name} = $walChange->oldkeys->keyvalues[$id];
            }
        }

        return json_encode($ret);
    }

    private function publishMessage(Message $message, $walChange, bool $retry = false): void
    {
        $routingKey = sprintf('%s.%s.%s', $this->routingKeyPrefix, $walChange->schema, $walChange->table);

        $metadata = $message->getProperties();
        $hash = $metadata['headers']['hash'] ?? '';

        $logString = '{operation} message';
        $context = [
            'component' => Constant::MQ_COMPONENT_NAME,
            'routing_key' => $routingKey,
            'operation' => (!$retry ? 'Send' : 'Re-send'),
            'schema' => $walChange->schema,
            'table' => $walChange->table,
            'kind' => $walChange->kind,
            'hash' => $hash,
            'event_name' => sprintf('%s.%s.%s.%s', $walChange->schema, $walChange->table, $walChange->kind, $hash),
        ];

        if (!$retry) {
            $this->logger->info($logString, $context);
        } else {
            $this->logger->notice($logString, $context);
        }

        $this->mqPublisher->publish(
            self::MESSAGE_TYPE,
            $message,
            [
                'routing_key' => $routingKey,
            ]
        );
    }

    private function recoverConnection(): void
    {
        $this->logger->notice('Recovering connection to message broker', [
            'component' => Constant::MQ_COMPONENT_NAME,
        ]);

        sleep(rand(1,5));

        $mqConfig = $this->mqPublisher->getConfigForMessageType(self::MESSAGE_TYPE);
        $this->brokerFactory->recoverConnection($mqConfig['exchange'], $mqConfig['connection']);
    }
}
