<?php

namespace App\Command;

use App\Common\Constant;
use App\Service\FilterParser;
use App\Service\Publisher\PublisherInterface;
use App\Service\Slot;
use Google\Cloud\PubSub\PubSubClient;
use Psr\Log\LoggerInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Symfony\Component\Process\Exception\ProcessFailedException;

abstract class AbstractParserCommand extends Command
{
    /** @var Slot */
    protected $slotService;

    /** @var LoggerInterface */
    protected $logger;

    /** @var string */
    protected $filters;

    /** @var FilterParser */
    protected $filterParser;

    /** @var PublisherInterface */
    protected $publisher;

    /** @var SymfonyStyle */
    protected $io;

    public function __construct(
        Slot $slotService,
        LoggerInterface $logger,
        FilterParser $filterParser,
        PublisherInterface $publisher,
        string $filters,
        string $name = null
    ){
        parent::__construct($name);
        $this->slotService = $slotService;
        $this->logger = $logger;
        $this->filters = $filters;
        $this->filterParser = $filterParser;
        $this->publisher = $publisher;
    }

    public function initialize(InputInterface $input, OutputInterface $output)
    {
        $this->io = new SymfonyStyle($input, $output);
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $slotName = $this->slotService->getSlotName();

        $this->logger->notice('Started {component}', [
            'component' => Constant::MQ_COMPONENT_NAME,
            'slot' => $slotName,
        ]);

        $filters = $this->filterParser->parse($this->filters);

        // Trying to create a replication slot named dbz_wal_parser
        $createSlotProcess = $this->slotService->createSlot();
        if (!$createSlotProcess->isSuccessful()) {
            if (!preg_match(
                sprintf('/replication slot "%s" already exists/m', $slotName),
                $createSlotProcess->getErrorOutput()
            )) {
                $this->logger->critical($createSlotProcess->getErrorOutput(), [
                    'component' => Constant::MQ_COMPONENT_NAME,
                    'slot' => $slotName,
                ]);
                throw new ProcessFailedException($createSlotProcess);
            }
        }
        $this->io->success("slot $slotName created");
        // Listening
        $listenSlotProcess = $this->slotService->getListenSlotProcess();
        $this->logger->notice('Listening to slot ...', [
            'component' => Constant::MQ_COMPONENT_NAME,
            'slot' => $slotName,
        ]);

        $listenSlotProcess->start();
        $this->io->success("Listening to slot $slotName ... ");
        foreach ($listenSlotProcess as $type => $buffer) {
            if ($listenSlotProcess::OUT === $type) {
                $bufferParts = explode("\n", $buffer);
                $changes = [];
                foreach ($bufferParts as &$bufferPart) {
                    $bufferPartData = json_decode(trim($bufferPart), false);

                    if (!empty($bufferPartData->change)) {
                        foreach ($bufferPartData->change as &$change) {
                            $change->timestamp = $bufferPartData->timestamp;
                        }

                        $changes = array_merge(
                            $changes,
                            $bufferPartData->change
                        );

                        unset($change);
                    }
                }
                unset($bufferPart);

                $nbMessages = count($changes);
                if ($nbMessages > 0) {
                    $this->io->success("Receive $nbMessages messages from pg_wal ");
                    $this->logger->info('Receive messages from pg_wal', [
                        'component' => Constant::MQ_COMPONENT_NAME,
                        'slot' => $slotName,
                        'nb_messages' => $nbMessages,
                    ]);
                }
                foreach ($changes as $walChange) {
                    $jsonWalChange = json_encode($walChange);
                    $this->logger->debug('Build message from wal change', [
                        'component' => Constant::MQ_COMPONENT_NAME,
                        'slot' => $slotName,
                        'wal_change' => $jsonWalChange,
                    ]);

                    $hash = sha1($jsonWalChange);
                    if (!$this->filterParser->acceptedForDelivery($walChange, $filters)) {
                        $this->logger->info(sprintf('Not accepted for delivery %s', $hash));
                        $this->io->success(sprintf('Not accepted for delivery %s', $hash));
                        continue;
                    }
                    $this->io->success('message to be published');
                    try {
                        $this->publisher->publish($walChange);
                        $this->io->success('message published.');
                    } catch (\Exception $exception) {
                        $this->logger->error($exception->getMessage(), [
                            'component' => Constant::MQ_COMPONENT_NAME,
                            'slot' => $slotName,
                            'exception_class' => get_class($exception),
                        ]);
                        continue;
                    }
                }
            }
        }

        return 0;
    }
}
