<?php

namespace App\Swarrot;

use PhpAmqpLib\Channel\AMQPChannel;
use PhpAmqpLib\Connection\AMQPSSLConnection;
use PhpAmqpLib\Connection\AMQPStreamConnection;
use Swarrot\Broker\MessagePublisher\PhpAmqpLibMessagePublisher;
use Swarrot\SwarrotBundle\Broker\AmqpLibFactory;

class DbzCustomProviderFactory extends AmqpLibFactory
{
    const HEARTBEAT = 20;
    const READ_WRITE_TIMEOUT = 40;

    /**
     * Return the AMQPChannel of the given connection.
     *
     * @param string $connection
     *
     * @return AMQPChannel
     */
    public function getChannel($connection)
    {
        if (isset($this->channels[$connection]) && !is_null($this->channels[$connection]->getConnection())) {
            return $this->channels[$connection];
        }

        if (!isset($this->connections[$connection])) {
            throw new \InvalidArgumentException(sprintf(
                'Unknown connection "%s". Available: [%s]',
                $connection,
                implode(', ', array_keys($this->connections))
            ));
        }

        if (!isset($this->channels[$connection])) {
            $this->channels[$connection] = [];
        }

        if (isset($this->connections[$connection]['ssl']) && $this->connections[$connection]['ssl']) {
            if (empty($this->connections[$connection]['ssl_options'])) {
                $ssl_opts = [
                    'verify_peer' => true,
                ];
            } else {
                $ssl_opts = [];
                foreach ($this->connections[$connection]['ssl_options'] as $key => $value) {
                    if (!empty($value)) {
                        $ssl_opts[$key] = $value;
                    }
                }
            }

            $conn = new AMQPSSLConnection(
                $this->connections[$connection]['host'],
                $this->connections[$connection]['port'],
                $this->connections[$connection]['login'],
                $this->connections[$connection]['password'],
                $this->connections[$connection]['vhost'],
                $ssl_opts,
                [
                    'heartbeat' => self::HEARTBEAT,
                ]
            );
        } else {
            $conn = new AMQPStreamConnection(
                $this->connections[$connection]['host'],
                $this->connections[$connection]['port'],
                $this->connections[$connection]['login'],
                $this->connections[$connection]['password'],
                $this->connections[$connection]['vhost'],
                false,
                'AMQPLAIN',
                null,
                'en_US',
                3.0,
                self::READ_WRITE_TIMEOUT,
                null,
                true,
                self::HEARTBEAT
            );
        }

        $this->channels[$connection] = $conn->channel();

        return $this->channels[$connection];
    }

    /**
     * {@inheritdoc}
     */
    public function getMessagePublisher($name, $connection)
    {
        if (!isset($this->messagePublishers[$connection][$name])) {
            if (!isset($this->messagePublishers[$connection])) {
                $this->messagePublishers[$connection] = [];
            }

            $channel = $this->getChannel($connection);

            $this->messagePublishers[$connection][$name] = new PhpAmqpLibMessagePublisher($channel, $name, true);
        }

        return $this->messagePublishers[$connection][$name];
    }

    /**
     * @param $name
     * @param $connection
     */
    public function recoverConnection($name, $connection)
    {
        $this->getChannel($connection)->getConnection()->reconnect();

        if (isset($this->messagePublishers[$connection][$name])) {
            unset($this->messagePublishers[$connection][$name]);
        }
        if (isset($this->channels[$connection])) {
            unset($this->channels[$connection]);
        }
    }
}
