<?php

namespace App\Command;

use App\Common\Constant;
use App\Service\FilterParser;
use App\Service\Slot;
use App\Swarrot\DbzCustomProviderFactory;
use PhpAmqpLib\Message\AMQPMessage;
use Psr\Log\LoggerInterface;
use Swarrot\Broker\Message;
use Swarrot\SwarrotBundle\Broker\AmqpLibFactory;
use Swarrot\SwarrotBundle\Broker\Publisher;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Process\Exception\ProcessFailedException;

class WalParserCommand extends Command
{
    private const MESSAGE_TYPE = 'event_producer';

    /**
     * @var Publisher
     */
    private $mqPublisher;

    /**
     * @var LoggerInterface
     */
    private $logger;

    /**
     * @var AmqpLibFactory
     */
    private $brokerFactory;

    /**
     * @var Slot
     */
    private $slotService;

    /**
     * @var FilterParser
     */
    private $filterParser;

    /**
     * @var array
     */
    private $filters;

    /**
     * @var string
     */
    private $routingKeyPrefix;

    public function __construct(
        Publisher $mqPublisher,
        LoggerInterface $logger,
        DbzCustomProviderFactory $brokerFactory,
        Slot $slotService,
        FilterParser $filterParser,
        string $filters,
        string $routingKeyPrefix
    ) {
        $this->mqPublisher = $mqPublisher;
        $this->logger = $logger;
        $this->brokerFactory = $brokerFactory;
        $this->slotService = $slotService;
        $this->filterParser = $filterParser;
        $this->routingKeyPrefix = $routingKeyPrefix;
        $this->filters = $filters;

        parent::__construct();
    }

    /**
     * {@inheritdoc}
     */
    protected function configure()
    {
        $this
            ->setName('pgs:wal:parser')
            ->setDescription('Create postgresql replication slot, then listen and parse incoming datas.')
            ->addOption(
                'file',
                'f',
                InputOption::VALUE_REQUIRED,
                'Configuration file'
            )
        ;
    }

    /**
     * {@inheritdoc}
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $slotName = $this->slotService->getSlotName();

        $this->logger->notice('Started {component}', [
            'component' => Constant::MQ_COMPONENT_NAME,
            'slot' => $slotName,
        ]);

        $filters = $this->filterParser->parse($this->filters);

        // Trying to create a replication slot named dbz_wal_parser
        $createSlotProcess = $this->slotService->createSlot();
        if (!$createSlotProcess->isSuccessful()) {
            if (!preg_match(
                sprintf('/replication slot "%s" already exists/m', $slotName),
                $createSlotProcess->getErrorOutput()
            )) {
                $this->logger->critical($createSlotProcess->getErrorOutput(), [
                    'component' => Constant::MQ_COMPONENT_NAME,
                    'slot' => $slotName,
                ]);
                throw new ProcessFailedException($createSlotProcess);
            }
        }

        // Listening
        $listenSlotProcess = $this->slotService->getListenSlotProcess();
        $this->logger->notice('Listening to slot ...', [
            'component' => Constant::MQ_COMPONENT_NAME,
            'slot' => $slotName,
        ]);

        $listenSlotProcess->start();

        foreach ($listenSlotProcess as $type => $buffer) {
            if ($listenSlotProcess::OUT === $type) {
                $bufferParts = explode("\n", $buffer);
                $changes = [];
                foreach ($bufferParts as &$bufferPart) {
                    $bufferPartData = json_decode(trim($bufferPart), false);

                    if (!empty($bufferPartData->change)) {
                        foreach ($bufferPartData->change as &$change) {
                            $change->timestamp = $bufferPartData->timestamp;
                        }

                        $changes = array_merge(
                            $changes,
                            $bufferPartData->change
                        );

                        unset($change);
                    }
                }
                unset($bufferPart);

                $nbMessages = count($changes);
                if ($nbMessages > 0) {
                    $this->logger->info('Receive messages from pg_wal', [
                        'component' => Constant::MQ_COMPONENT_NAME,
                        'slot' => $slotName,
                        'nb_messages' => $nbMessages,
                    ]);
                }
                foreach ($changes as $walChange) {
                    $jsonWalChange = json_encode($walChange);
                    $this->logger->debug('Build message from wal change', [
                        'component' => Constant::MQ_COMPONENT_NAME,
                        'slot' => $slotName,
                        'wal_change' => $jsonWalChange,
                    ]);
                    $body = $this->createMessageFromWalChange($walChange);
                    $hash = sha1($jsonWalChange);

                    if (!$this->acceptedForDelivery($body, $filters)) {
                        $this->logger->info(sprintf('Not accepted for delivery %s', $hash));
                        continue;
                    }

                    $message = new Message(json_encode($body), [
                        'headers' => [
                            'hash' => $hash,
                        ],
                        'delivery_mode' => AMQPMessage::DELIVERY_MODE_PERSISTENT
                    ]);

                    try {
                        $this->publish($message, $walChange);
                    } catch (\Exception $publishException) {
                        $this->logger->warning($publishException->getMessage(), [
                            'component' => Constant::MQ_COMPONENT_NAME,
                            'slot' => $slotName,
                            'exception_class' => get_class($publishException),
                        ]);

                        $this->recoverConnection();

                        try {
                            $this->publish($message, $walChange, true);
                        } catch (\Exception $republishException) {
                            $this->logger->critical($republishException->getMessage(), [
                                'component' => Constant::MQ_COMPONENT_NAME,
                                'slot' => $slotName,
                                'exception_class' => get_class($republishException),
                            ]);
                        }
                    }
                }
            }
        }

        return 0;
    }

    private function acceptedForDelivery(\stdClass $body, array $filters)
    {
        if (empty($filters)) {
            return true;
        }

        if ($body->type == 'delete') {
            return true;
        }

        $new = $body->new;

        foreach ($filters as $name => $filter) {
            if (!isset($new->{$name})) {
                continue;
            }

            $walValue = $new->{$name};

            // dates
            if ($filter['value'] instanceof \DateTime) {
                $walValue = new \DateTime($new->{$name});
            }

            switch ($filter['operator']) {
                case '=':
                    if ($filter['value'] != $walValue) {
                        return false;
                    }
                    break;
                case '<=':
                    if ($walValue > $filter['value']) {
                        return false;
                    }
                    break;
                case '>=':
                    if ($walValue < $filter['value']) {
                        return false;
                    }
                    break;
                case '<':
                    if ($walValue >= $filter['value']) {
                        return false;
                    }
                    break;
                case '>':
                    if ($walValue <= $filter['value']) {
                        return false;
                    }
                    break;
            }

            return true;
        }
    }

    /**
     * generate a message to send to the MQ
     *
     * @param \stdClass $walChange
     *
     * @return \stdClass
     */
    private function createMessageFromWalChange(\stdClass $walChange): \stdClass
    {
        $ret = new \stdClass();
        $ret->kind = $walChange->kind;
        $ret->schema = $walChange->schema;
        $ret->table = $walChange->table;
        $ret->wal_timestamp = (new \DateTime($walChange->timestamp))->format('c');
        $ret->parse_timestamp = (new \DateTime())->format('c');

        if ($walChange->kind === 'insert' || $walChange->kind === 'update') {
            $ret->new = new \stdClass();

            foreach ($walChange->columnnames as $id => $name) {
                $ret->new->{$name} = $walChange->columnvalues[$id];
            }
        }

        if ($walChange->kind === 'delete' || $walChange->kind === 'update') {
            $ret->old = new \stdClass();

            foreach ($walChange->oldkeys->keynames as $id => $name) {
                $ret->old->{$name} = $walChange->oldkeys->keyvalues[$id];
            }
        }

        return $ret;
    }

    /**
     * @param Message $message
     * @param $walChange
     * @param bool $retry
     */
    private function publish(Message $message, $walChange, bool $retry = false): void
    {
        $routingKey = sprintf('%s.%s.%s', $this->routingKeyPrefix, $walChange->schema, $walChange->table);

        $metadata = $message->getProperties();
        $hash = $metadata['headers']['hash'] ?? '';

        $logString = '{operation} message';
        $context = [
            'component' => Constant::MQ_COMPONENT_NAME,
            'routing_key' => $routingKey,
            'operation' => (!$retry ? 'Send' : 'Re-send'),
            'schema' => $walChange->schema,
            'table' => $walChange->table,
            'kind' => $walChange->kind,
            'hash' => $hash,
            'event_name' => sprintf('%s.%s.%s.%s', $walChange->schema, $walChange->table, $walChange->kind, $hash),
        ];

        if (!$retry) {
            $this->logger->info($logString, $context);
        } else {
            $this->logger->notice($logString, $context);
        }

        $this->mqPublisher->publish(
            self::MESSAGE_TYPE,
            $message,
            [
                'routing_key' => $routingKey,
            ]
        );
    }

    private function recoverConnection(): void
    {
        $this->logger->notice('Recovering connection to message broker', [
            'component' => Constant::MQ_COMPONENT_NAME,
        ]);

        $mqConfig = $this->mqPublisher->getConfigForMessageType(self::MESSAGE_TYPE);
        $this->brokerFactory->recoverConnection($mqConfig['exchange'], $mqConfig['connection']);
    }
}
