<?php

namespace Orms\DataProvider;

abstract class GenericDataProvider
{
    /**
     * Connection - can be any data provider from a database to a csv
     * file or a rest service
     */

    protected $_connection;

    /**
     * Store an external context to be accessible in model
     *
     * @var object
     */
    
    protected static $externalContextInstance;

    /**
     * Logger instance
     * 
     * @var type
     */
    
    protected static $_logger;

    /**
     * Metadata object
     * 
     * @var array of meta objects
     */
    
    protected static $_meta = array();

    /**
     * Constructor - create the connection to the service used.
     *
     * @param Connection primary keys of the data provided
     * @param mixed $options
     * @param $logger
     * @param $externalContext
     */

    public function __construct(
        $connection,
        $options = null,
        $logger = null,
        $externalContext = null
    )
    {
        $this->_connection               = $connection;
        static::$_logger                 = $logger;
        static::$externalContextInstance = $externalContext;

        $this->setup($options);
    }

    /**
     * Return connection Handler
     * @return array
     */
    
    public function getConnection()
    {
        return $this->_connection->getConnection();
    }

    /**
     * Return external context
     * @return mixed
     */
    
    public static function getExternalContext()
    {
        return static::$externalContextInstance;
    }

    /**
     * Remove connection reference
     */
    public function __destruct()
    {
        unset($this->_connection);
    }

    /**
     * @param mixed $options
     */

    abstract public function setup($options);

    /**
     * @param array $definition
     * @param array $values
     *
     * @return Array (value of the object after the insert)
     */

    abstract public function insert(
        Array $definition,
        Array $values
    );

    /*
      * @param array $definition
      * @param array $values
      *
      * @return Array (value of the object after the insert)
      */

    abstract public function update(
        Array $definition,
        Array $values
    );

    /*
      * @param array $definition
      * @param array $values
      *
      * @return boolean (wether the delete was successful or not)
      */

    abstract public function delete(
        Array $definition,
        Array $values
    );

    /**
     * getter / Query composer
     *
     * @param array $definition
     * @param array $values
     *
     * Toute la composition des query composer est statique car on se sert de
     * l'héritage pour éviter de répéter des données dans les tableaux de
     * config.
     */

    public function get(Array $params)
    {
        return static::getComposer($params);
    }

    /**
     * get last execution meta data
     *
     * @return object
     */

    public function getMetaData()
    {
        return static::$_meta[get_called_class()];
    }

    /**
     * Paramètres de base d'une méthode get
     *
     * Ce tableau fixe les paramètres de base d'une requête get à moins d'une
     * réécriture dans une classe fille. Il n'est pas nécessaire de réécrire
     * tous les paramètres.
     *
     * @see get()
     *
     * @var    array
     */

    static public $_getBaseParams =
        array
        (
            'countOnly'   => false,
            'offset'      => 0,
            'limit'       => null,
            'orderType'   => null,
            'orderAsc'    => true,
    
            '_onlyFields' => null,
        );

    /**
     * Paramètres pour filtrer les données d'une méthode get
     *
     * Ce tableau définit comment les données seront filtrées d'une requete get.
     * parent_id ou parent_path sont généralement settés par le paramettre
     * $parentParam de la méthode get, même si un filtrage est toujours possible
     * en sus. Il est tout à fait possible qu'un filtre nécessite une jointure.
     *
     * Si l'on veut qu'une requète get soit filtrée en permanence, il faut
     * définir le filtre par défaut avec une valeur non nulle.
     *
     * @var array
     */

    static public $_getFilterParams = array();

    /**
     * La grande méthode du get, appelée par les méthodes de la classe
     * jDbObject et toute sa déscendance. Cette méthode utilise les
     * tableaux de configuration définis ci-dessus
     *
     * @param array   $params
     *
     * @return integer (count) or ArrayObject
     *
     * @author Yannick Le Guédart
     */

    static protected function getComposer($params)
    {
        return;
    }

    /**
     * Return the logger instance
     * @return type
     */
    protected static function getLogger()
    {
        return static::$_logger;
    }
}
