<?php

namespace Orms\Connection;

use Orms\Connection\SqlInterface;
use Orms\Exception\DatabaseException;
use Orms\Statement\PostgresqlStatement;

/**
 * Connection à Postgresql
 *
 * @author Xavier HAUSHERR <xavier.hausherr@ebuzzing.com>
 */
class Postgresql implements SqlInterface
{
    /**
     * PG Connection handler
     * @var ressource
     */
    protected $connection;

    /**
     * Ouvre une nouvelle connection Postgresql
     *
     * @param array $options
     * @throws \Orms\Exception\Exception
     */
    public function __construct($options = array())
    {
        $dsn = sprintf(
                'host=%s dbname=%s user=%s password=%s',
                $options['host'],
                $options['dbname'],
                $options['username'],
                $options['password']
            );

        if(isset($options['port']))
        {
            $dsn.= sprintf(' port=%s', $options['port']);
        }

        $this->connection = @pg_connect($dsn);

        if(false === $this->connection)
        {
            throw new DatabaseException(
                    'Unable to connect to PostgreSQL server'
                );
        }
    }

    /**
     * {@inheritDoc}
     */
    public function ping()
    {
        return pg_ping($this->connection);
    }

    /**
     * {@inheritDoc}
     */
    public function quote($data)
    {
        return pg_escape_literal($this->connection, $data);
    }

    /**
     * {@inheritDoc}
     */
    public function query($sql)
    {
        $result = @pg_query($this->connection, $sql);

        if(false !== $result)
        {
            return new PostgresqlStatement($result);
        }
        else
        {
            return false;
        }
    }

    /**
     * {@inheritDoc}
     */
    public function prepare($sql)
    {
        $id = microtime(true);

        $result = pg_prepare($this->connection, $id, $sql);

        if(false !== $result)
        {
            return new PostgresqlStatement($result, $id, $this->connection);
        }
        else
        {
            return false;
        }
    }

    /**
     * {@inheritDoc}
     */
    public function exec($sql)
    {
        $result = @pg_query($this->connection, $sql);

        if(false !== $result)
        {
            return pg_affected_rows($result);
        }
        else
        {
            return false;
        }
    }

    /**
     * {@inheritDoc}
     */
    public function errorCode()
    {
        return null;
    }

    /**
     * {@inheritDoc}
     */
    public function errorInfo()
    {
        return pg_last_error($this->connection);
    }

    /**
     * {@inheritDoc}
     */
    public function __destruct()
    {
        return @pg_close($this->connection);
    }
}