<?php

namespace Orms\Statement;

use Orms\Exception\SqlExceptionInterface;

/**
 * MysqlStatement
 * 
 * Représente une requête MySQL préparée et, une fois exécutée,
 * le jeu de résultats associé.
 *
 * @author Xavier HAUSHERR <xavier.hausherr@ebuzzing.com>
 */
class MysqlStatement extends Statement implements SqlExceptionInterface
{
    /**
     * {@inheritDoc}
     */
    public function __destruct()
    {
        if(is_resource($this->resource))
        {
            return mysqli_free_result($this->resource);
        }
    }

    /**
     * {@inheritDoc}
     */
    public function errorCode()
    {
        return mysqli_errno($this->connectionRessource);
    }

    /**
     * {@inheritDoc}
     */
    public function errorInfo()
    {
        return mysqli_error($this->connectionRessource);
    }

    /**
     * {@inheritDoc}
     */
    public function execute($params)
    {
        $types = array();
        $paramsRef = array();
        $bindParam = array($this->resource);

        foreach($params as $key => $param)
        {
            switch(gettype($param))
            {
                case 'integer':
                    $types[] = 'i';
                    break;

                case 'double':
                    $types[] = 'd';
                    break;

                default:
                    $types[] = 's';
                    break;
            }

            // Var must be passed as ref
            $paramsRef[] = &$params[$key];
        }

        $bindParam[] = implode('', $types);

        // Bind param
        call_user_func_array(
            'mysqli_stmt_bind_param',
            array_merge($bindParam, $paramsRef)
        );

        return @mysqli_stmt_execute($this->resource);
    }

    /**
     * {@inheritDoc}
     */
    public function fetch($style = self::FETCH_ASSOC, $index = null)
    {
        if(!is_null($index))
        {
            mysqli_data_seek($this->resource, $index);
        }

        switch($style)
        {
            case self::FETCH_OBJ:
                return mysqli_fetch_object($this->resource);

            case self::FETCH_NUM:
                return mysqli_fetch_row($this->resource);

            case self::FETCH_ASSOC:
            default:
                return mysqli_fetch_assoc($this->resource);
        }
    }

    /**
     * {@inheritDoc}
     */
    public function lastInsertId()
    {
        return mysqli_stmt_insert_id($this->resource);
    }

    /**
     * {@inheritDoc}
     */
    public function rowCount()
    {
        return mysqli_num_rows($this->resource);
    }
}
