<?php

namespace Orms\Type;

/**
 * Class for hstore typing
 */

class Hstore extends Type implements TypeInterface
{
    /**
     * The exploded hashstore
     * @access protected
     * @var object
     */

    protected $_value = array();

    /**
     * The string to be escaped
     * @var object
     */

    static private $_escapeArray =
        array(
            '"' => '{_[DQUOT]_}',
            "\n" => '{_[BR]_}',
            '\\' => '{_[BS]_}'
        );

    /**
     * sets the value of the object after checking
     *
     * @param mixed $value
     *
     * @return object
     */

    public function setValue($value)
    {
        parent::setValue($value);

        $this->_decode($value);
    }

    /**
     * Gets an usable value of the object
     *
     * @return mixed
     */

    public function getValue()
    {
        return $this;
    }

    /**
     * Checks if the value provided is valid
     *
     * @param mixed $value
     *
     * @returns boolean
     */

    public function isValid($value)
    {
        // $value must be a string || an array

        if (is_array($value))
        {
            return true;
        }

        if (is_string($value))
        {        
            // If the string is empty, the $value is valid
    
            if ('' === $value)
            {
                return true;
            }
    
            // validation de la chaine. On considère pour l'instant que la
            // chaine du store est valide si elle a au moins une clef/valeur
    
            preg_match_all('/"(.+?)"=>"(.*?)"(,|$)/', $value, $m);
    
            return (count($m[0]) !== 0);
        }
        
        return false;
    }

    /**
     * Decode a string to an array
     *
     * @param string $string
     *
     * @return void
     */

    private function _decode($value)
    {
        if (is_array($value))
        {
            $this->_value = $value;
        }
        else // it's a string
        {
            $this->_value = array();
            
            if ($value !== '')
            {
                preg_match_all('/"(.+?)"=>"(.*?)"(,|$)/', $value, $m);
    
                $nbPairs = count($m[0]);
    
                for ($i = 0; $i < $nbPairs; $i++)
                {
                    $this->_value[$this->_decodeField($m[1][$i])] =
                            $this->_decodeField($m[2][$i]);
                }
            }
        }
    }

    /**
     * __unset
     *
     * The magic unsetter (including constraint on attributes)
     *
     * @param string $name
     * @param mixed  $value
     *
     * @return void
     */

    public function __unset($name)
    {
        if (is_string($name))
        {
            unset($this->_value[$name]);
        }
    }

    /**
     * __set
     *
     * The magic setter (including constraint on attributes)
     *
     * @param string $name
     * @param mixed  $value
     *
     * @return void
     */

    public function __set($name, $value)
    {
        if (is_string($name))
        {
            $this->_value[$name] = $value;
        }
    }

    /**
     * __get
     *
     * The magic getter
     *
     * @param string $name
     *
     * @return mixed
     */

    public function __get($name)
    {
        if (is_string($name) && isset($this->_value[$name]))
        {
            return $this->_value[$name];
        }

        return null;
    }

    /**
     * __isset
     *
     * @param string $name
     *
     * @return mixed
     */

    public function __isset($name)
    {
        return array_key_exists($name, $this->_value);
    }

    /**
     * __toString
     *
     * Returns the string version of the hstore
     *
     * @return string
     */

    public function __toString()
    {
        $splittedItemsArray = array();

        foreach ($this->_value as $k => $v)
        {
            $splittedItemsArray[] =
                    '"' .
                    $this->_encodeField($k) .
                    '"=>"' .
                    $this->_encodeField($v) .
                    '"';
        }

        return implode(',', $splittedItemsArray);
    }

    /**
     * Encode an hstore string
     *
     * @return string
     */

    public function _encodeField($string)
    {
        return
                str_replace(
                    array_keys(self::$_escapeArray),
                    array_values(self::$_escapeArray),
                    $string
                );
    }


    /**
     * Decode an hstore string
     *
     * @return string
     */

    static public function _decodeField($string)
    {
        return
                str_replace(
                    array_values(self::$_escapeArray),
                    array_keys(self::$_escapeArray),
                    $string
                );
    }

    /**
     * toArray
     *
     * Returns the array version of the hstore
     *
     * @return string
     */

    public function toArray()
    {
        return $this->_value;
    }

}