<?php

namespace Orms\Connection;

use Orms\Connection\SqlInterface;
use Orms\Exception\DatabaseException;
use Orms\Statement\MssqlStatement;

/**
 * Connection à Mssql
 *
 * @author Yannick Le Guédart <yannick@overblog.com>
 */

class Mssql implements SqlInterface
{
    /**
     * Mssql Connection handler
     * @var ressource
     */

    protected $connection;

    /**
     * Ouvre une nouvelle connection Mssql
     *
     * @param  array $options
     *
     * @throws DatabaseException
     */

    public function __construct($options = array())
    {
        $this->connection =
            @mssql_connect(
                $options['host'] .
                    (isset($options['port']) ? (' : ' . $options['port']) : ''),
                $options['username'],
                $options['password']
            );

        if (false === $this->connection)
        {
            throw new DatabaseException('Unable to connect to MsSQL server');
        }

        if (false === mssql_select_db($options['dbname'], $this->connection))
        {
            throw new DatabaseException(
                'Invalid database [' . $options['dbname'] . ']'
            );
        }
    }

    /**
     * {@inheritDoc}
     */

    public function ping()
    {
        return (false !== $this->query('SELECT 1;'));
    }

    /**
     * {@inheritDoc}
     */

    public function quote($data)
    {
        return mb_str_replace("'", "''", $data);
    }

    /**
     * {@inheritDoc}
     */

    public function query($sql)
    {
        $result = @mssql_query($sql, $this->connection);

        if (false !== $result)
        {
            return new MssqlStatement($result);
        }
        else
        {
            return false;
        }
    }

    /**
     * {@inheritDoc}
     */

    public function prepare($sql)
    {
        throw new DatabaseException("MsSQL can't use prepare.");
    }

    /**
     * {@inheritDoc}
     */

    public function exec($sql)
    {
        $result = @mssql_query($sql, $this->connection);

        if (false !== $result)
        {
            return mssql_rows_affected($result);
        }
        else
        {
            return false;
        }
    }

    /**
     * {@inheritDoc}
     */

    public function errorCode()
    {
        return null;
    }

    /**
     * {@inheritDoc}
     */

    public function errorInfo()
    {
        return mssql_get_last_message();
    }

    /**
     * {@inheritDoc}
     */

    public function __destruct()
    {
        return @mssql_close($this->connection);
    }
}