<?php

namespace Orms\Collections;

use Orms\DataProvider\GenericDataProvider;
use Orms\Statement\Statement;

/**
 * DataObjectCollection return an array Iterable of DataObject
 *
 * @author Xavier HAUSHERR <xavier.hausherr@ebuzzing.com>
 */
class DataObjectCollection implements \Iterator, \Countable, \ArrayAccess
{
    /**
     * Data Provider instance
     * @var \Orms\DataProvider\GenericDataProvider
     */
    protected $provider;

    /**
     * Query Statement
     * @var \Orms\Statement\Statement
     */
    protected $statement;

    /**
     * ClassName to instanciate
     * @var string
     */
    protected $className;

    /**
     * Array of links configuration
     * @var array
     */
    protected $links = [];

    /**
     * Init Iterator
     * @param \Orms\DataProvider\GenericDataProvider $provider
     * @param \Orms\Statement\Statement $statement
     * @param string $className
     * @param array $links
     */
    public function __construct(GenericDataProvider $provider, Statement $statement, $className, Array $links)
    {
        $this->provider = $provider;
        $this->statement = $statement;
        $this->className = $className;
        $this->links = $links;
    }

    /**
     * Retourne l'élément courant
     * @return \Orms\Object\DataObject
     */
    public function current()
    {
        return new $this->className(
                $this->provider,
                new \ArrayObject($this->statement->current()),
                $this->links
            );
    }

    /**
     * Retourne la clé de l'élément courant
     * @return string
     */
    public function key()
    {
        return $this->statement->key();
    }

    /**
     * Se déplace sur l'élément suivant
     */
    public function next()
    {
        $this->statement->next();
    }

    /**
     * Replace l'itérateur sur le premier élément
     */
    public function rewind()
    {
        $this->statement->rewind();
    }

    /**
     * Vérifie si la position courante est valide
     * @return boolean
     */
    public function valid()
    {
        return $this->statement->valid();
    }

    /**
     * Compte le nombre d'éléments
     * @return int
     */
    public function count()
    {
        return $this->statement->count();
    }

    /**
     * Indique si une position existe dans un tableau
     *
     * @param int $offset
     * @return boolean
     */
    public function offsetExists($offset)
    {
        return $offset < $this->count();
    }

    /**
     * Retourne la valeur à la position donnée.
     *
     * @param int $offset
     * @return \Orms\Object\DataObject
     */
    public function offsetGet($offset)
    {
        return new $this->className(
                $this->provider,
                new \ArrayObject(
                        $this->statement->fetch(Statement::FETCH_ASSOC, $offset)
                    ),
                $this->links
            );
    }

    /**
     * Assigne une valeur à une position donnée.
     *
     * @deprecated
     * @param int $offset
     * @param mixed $value
     * @throws \LogicException
     */
    public function offsetSet($offset, $value)
    {
        throw new \LogicException(
                'You cannot add an element to a DataObjectCollection');
    }

    /**
     * Supprime une position.
     *
     * @deprecated
     * @param int $offset
     * @throws \LogicException
     */
    public function offsetUnset($offset)
    {
        throw new \LogicException(
                'You cannot unset an element of DataObjectCollection');
    }
}
