<?php

namespace Orms\DataProvider;

use Orms\Cache\CacheableInterface;
use Orms\Object\DataObject;
use Orms\Logger\LoggerInterface;
use Orms\Cache\CacheInterface;
use Orms\Cache\CacheKey;
use Orms\EventDispatcher\EventDispatcherInterface;

abstract class GenericDataProvider
{
    const ORDER_ASC  = true;
    const ORDER_DESC = false;

    /**
     * Connection - can be any data provider from a database to a csv
     * file || a rest service
     */

    protected $_connection;

    /**
     * Store an external context to be accessible in model
     *
     * @var object
     */

    protected static $externalContextInstance;

    /**
     * Logger instance
     *
     * @var LoggerInterface
     */

    protected static $_logger;

    /**
     * Cache instance
     * @var CacheInterface
     */

    protected static $_cache;

    /**
     * Dispatcher instance
     * @var EventDispatcherInterface
     */

    protected $_dispatcher;

    /**
     * Links between object for cache clean
     * @var array
     */

    protected $cacheLinks = array();

    /**
     * Metadata object
     *
     * @var array of meta objects
     */

    protected static $_meta = array();

    /**
     * Constructor - create the connection to the service used.
     *
     * @param Connection primary keys of the data provided
     * @param mixed $options
     * @param $externalContext
     * @param LoggerInterface $logger
     * @param EventDispatcherInterface $dispatcher
     * @param CacheInterface $cache
     */

    public function __construct(
        $connection,
        $options = null,
        $externalContext = null,
        LoggerInterface $logger = null,
        CacheInterface $cache = null,
        EventDispatcherInterface $dispatcher = null,
        $cacheLinks = array()
    )
    {
        $this->_connection               = $connection;
        static::$externalContextInstance = $externalContext;
        static::$_logger                 = $logger;
        static::$_cache                  = $cache;
        $this->_dispatcher               = $dispatcher;
        $this->cacheLinks                = $cacheLinks;

        $this->setup($options);
    }

    /**
     * Return connection Handler
     * @return array
     */

    public function getConnection()
    {
        return $this->_connection->getConnection();
    }

    /**
     * Return external context
     * @return mixed
     */

    public static function getExternalContext()
    {
        return static::$externalContextInstance;
    }

    /**
     * Determine if object can be added to cache
     * @return boolean
     */
    public function isCacheable()
    {
        return
            !is_null(static::$_cache) &&
            $this instanceof CacheableInterface;
    }

    /**
     * Return cache key build with params
     * @param mixed $key
     * @return CacheKey
     */
    private function getCacheKey($key)
    {
        $param = array();

        // Remove private vars
        foreach($key as $k => $v)
        {
            if('_' !== substr($k, 0, 1))
            {
                $param[$k] = $v;
            }
        }

        return new CacheKey(
                $param[static::getCacheProperty()],
                get_class($this) . sha1(serialize($param))
            );
    }

    /**
     * Clean cache for linked DataProvider
     * @param \Orms\Object\DataObject $object
     */
    public function cacheCleanForLinkedObjects(DataObject $object)
    {
        if(
            !is_null(static::$_cache) &&
            isset($this->cacheLinks[get_class($object)])
        )
        {
            foreach($this->cacheLinks[get_class($object)] as $link)
            {
                try
                {
                    static::$_cache->clean(
                            new CacheKey(
                                $object->{$link::getCacheProperty()},
                                null
                            )
                        );
                }
                catch(\Exception $e)
                {
                    static::getLogger()->logException($e->getMessage());
                }
            }
        }
    }

    /**
     * Remove connection reference
     */
    public function __destruct()
    {
        unset($this->_connection);
    }

    /**
     * @param mixed $options
     */

    abstract public function setup($options);

    /**
     * @param array $definition
     * @param array $values
     *
     * @return Array (value of the object after the insert)
     */

    abstract public function insert(
        Array $definition,
        Array $values
    );

    /*
      * @param array $definition
      * @param array $values
      *
      * @return Array (value of the object after the insert)
      */

    abstract public function update(
        Array $definition,
        Array $values
    );

    /*
      * @param array $definition
      * @param array $values
      *
      * @return boolean (wether the delete was successful || not)
      */

    abstract public function delete(
        Array $definition,
        Array $values
    );

    /**
     * getter / Query composer
     *
     * @param array $definition
     * @param array $values
     *
     * Toute la composition des query composer est statique car on se sert de
     * l'héritage pour éviter de répéter des données dans les tableaux de
     * config.
     */

    public function get(Array $params)
    {
        $is_Cacheable = $this->isCacheable();

        // Read Cache
        if($is_Cacheable)
        {
            $key = $this->getCacheKey($params);

            try
            {
                if((list($result, $metadata) = static::$_cache->get($key)))
                {
                    // Set metadata
                    static::$_meta[get_called_class()] = $metadata;

                    return $result;
                }
            }
            catch(\Exception $e)
            {
                static::getLogger()->logException($e->getMessage());
            }
        }

        // Exec Query
        $result = static::getComposer($params);

        // Write Cache
        if($is_Cacheable)
        {
            try
            {
                static::$_cache->set($key, $result, $this->getMetaData());
            }
            catch(\Exception $e)
            {
                static::getLogger()->logException($e->getMessage());
            }
        }

        return $result;
    }

    /**
     * get last execution meta data
     *
     * @return object
     */

    public function getMetaData()
    {
        return static::$_meta[get_called_class()];
    }

    /**
     * Paramètres de base d'une méthode get
     *
     * Ce tableau fixe les paramètres de base d'une requête get à moins d'une
     * réécriture dans une classe fille. Il n'est pas nécessaire de réécrire
     * tous les paramètres.
     *
     * @see get()
     *
     * @var    array
     */

    static public $_getBaseParams =
        array
        (
            'countOnly'   => false,
            'offset'      => 0,
            'limit'       => null,
            'orderType'   => null,
            'orderAsc'    => true,

            // New way of handling orders

            'order'       => null,

            '_onlyFields' => null,
        );

    /**
     * Paramètres pour filtrer les données d'une méthode get
     *
     * Ce tableau définit comment les données seront filtrées d'une requete get.
     * parent_id ou parent_path sont généralement settés par le paramettre
     * $parentParam de la méthode get, même si un filtrage est toujours possible
     * en sus. Il est tout à fait possible qu'un filtre nécessite une jointure.
     *
     * Si l'on veut qu'une requète get soit filtrée en permanence, il faut
     * définir le filtre par défaut avec une valeur non nulle.
     *
     * @var array
     */

    static public $_getFilterParams = array();

    /**
     * La grande méthode du get, appelée par les méthodes de la classe
     * jDbObject et toute sa déscendance. Cette méthode utilise les
     * tableaux de configuration définis ci-dessus
     *
     * @param array   $params
     *
     * @return integer (count) || ArrayObject
     *
     * @author Yannick Le Guédart
     */

    static protected function getComposer($params)
    {
        return;
    }

    /**
     * Return the logger instance
     *
     * @return LoggerInterface
     */

    protected static function getLogger()
    {
        return static::$_logger;
    }

    /**
     * Notify Event Dispatcher of a DataObject write
     *
     * @param DataObject $object
     * @param int $eventType
     */

    public function notify(DataObject $object, $eventType)
    {
        if(!is_null($this->_dispatcher))
        {
            $this->_dispatcher->notify($object, $eventType);
        }
    }
}
