<?php

namespace Orms\Object;

use Orms\Exception\Exception;
use Orms\Tools\Inflector;
use Orms\Exception\NoRowsFoundException;
use Orms\Object\DataObjectDefinition;
use Orms\Type\Hstore;
use Orms\Type\PgArray;
use Orms\Type\Xml;

/**
 * DataObject - abstract class for data mapping and attribute accessing
 *
 * @abstract
 * @package Orms
 * @version $id$
 *
 * @copyright 2011 Overblog
 *
 * @author Yannick Le Guédart <yannick@overblog.com>
 *
 */

abstract class DataObject implements DataObjectDefinition, \Serializable
{
    /**
     * Data provider - can be any data provider from a database to a csv
     * file or a rest service
     */

    protected $_provider = null;

    /**
     * List of primary keys.
     * @var array
     */

    static protected $_pkFields;

    /**
     * Whether a primary key has been changed.
     * @var boolean
     */

    protected $_pkChanged = false;

    /**
     * List of columns/attributes.
     *
     * Each entry is represented this way :
     *
     * 'name' =>
     *         array(
     *             'type' => '...',
     *             'has_default' => '...' // Default handled by the data storage
     *             'default' => '...' // Default handled directly in php
     *         )
     *
     * @var array
     */

    static protected $_columns = array();

    /**
     * Contains the values received from the database, or the values to be sent
     * to it
     *
     * @var array
     */

    protected $_string_values = array();

    /**
     * Contains the complex objects of different values. Used for validation
     * and complex operations. These object are instantiated on demand.
     *
     * @var array
     */

    protected $_object_values = array();

    /**
     * The external attribute. It can be used to store more information than
     * the original data of the table from linked tables
     *
     * @var object
     */

    public $external = null;

    /**
     * Whether the object is savable or not.
     *
     * @var boolean
     */

    protected $_savable = true;

    /**
     * The class constructor
     *
     * $data contient soit une donnée numérique, qui correspond à l'id dans la
     * base, soit un objet ou un tableau à partir desquels sera construit
     * l'objet sans se soucier de la base.
     *
     * @param mixed $data
     * @param array $params
     *
     * @return void
     */

    public function __construct(
        $provider = null,
        $data = null,
        $links = null
    )
    {
        if (is_null($provider))
        {
            throw new Exception(
                'Argument 1 passed to Orms\Object\DataObject::__construct() ' .
                'must be an instance of Orms\DataProvider\GenericDataProvider, ' .
                'none given'
            );
        }

        // Initialisation

        $this->_init($provider, $links);

        // Maintenant, on va tenter de construire notre objet. Le permier cas,
        // c'est d'avoir un data non défini.

        if (! is_null($data))
        {
            // Si $data est numérique ou chaine, alors on va recherche les
            // données en base à partir de la clef primaire si on n'en a qu'une
            // seule.

            if (
                (is_numeric($data) or is_string($data))
                and (count(static::$_pkFields) === 1)
            )
            {
                $rs =
                        $this->_provider->get(
                            array(
                                 static::$_pkFields[0] => $data
                            )
                        );

                if ($rs->count() === 0)
                {
                    throw new NoRowsFoundException(
                        "Can't find " . get_called_class() . ' object in data ' .
                        'provider with the primary key [' . $data . ']'
                    );
                }

                $this->setFrom($rs->offsetGet(0));
            }
            elseif (is_array($data))
            {
                // Là on un tableau qu'on pense contenir les clefs nécessaire à
                // récupérer un unique enregistrement. On fait la requete pour
                // aller chercher jusqu'à deux, et on réponf OK si on n'en
                // récupère qu'un

                $data['limit'] = 2;

                $rs =
                        $this->_provider->get(
                            $data
                        );

                if ($rs->count() === 0)
                {
                    throw new NoRowsFoundException(
                        "Can't find " . get_called_class() . ' object in data' .
                        ' provider with the params [' .
                        print_r($data, true) . ']'
                    );
                }
                elseif ($rs->count() > 1)
                {
                    throw new Exception(
                        "Found more than one " . get_called_class() .
                        ' objects in data provider with the params [' .
                        print_r($data, true) . ']'
                    );
                }

                $this->setFrom($rs->offsetGet(0));
            }
            elseif (is_object($data))
            {
                $this->setFrom($data);
            }
            else
            {
                throw new Exception(
                    'Invalid data type [' . gettype($data) . ']'
                );
            }
        }
    }

    /**
     * récupération de la définition des clefs primaires / colonnes
     *
     * @return array
     */

    public function getDefinition()
    {
        return array(
            'pkFields' => static::$_pkFields,
            'columns' => static::$_columns
        );
    }

    /**
     * Store an external context to be accessible in model
     * @return mixed
     */
    public function getExternalContext()
    {
        $provider = $this->_provider;
        return $provider::getExternalContext();
    }

    /**
     * Clean connection
     */
    public function __destruct()
    {
        unset($this->_provider);
    }

    /**
     * Initialize the object before the constructor.
     *
     * @return void
     */

    protected $_init = false;

    protected final function _init(
        $provider,
        Array $links = null
    )
    {
        // sauvegarde du provider

        $this->_provider = $provider;

        // Sauvegarde des link

        $this->_linkedElement = $links;

        // On Crée les colonnes de valeur.

        foreach (static::$_columns as $k => $v)
        {
            $this->_object_values[$k] = null;
            $this->_string_values[$k] = null;

            if (isset($v['default']))
            {
                $this->{$k} = $v['default'];
            }
        }

        // On instancie l'objet external pour toute donnée externe à la
        // dao qu'on voudrait trimballer avec l'objet :

        $this->external = new \StdClass();

        // Puis on note que l'initialisation a eu lieu pour cet objet

        $this->_init = true;
    }

    /**
     * The magic setter
     *
     * @param string $name
     * @param mixed  $value
     *
     * @return void
     */

    public function __set($name, $value)
    {
        // Si la promriété n'existe pas, on lève une exception

        if (!isset(static::$_columns[$name]))
        {
            throw new Exception(
                'Property [' . $name . '] does not exist.'
            );
        }

        // Et donc là on arrive et on met à jour la propriété

        if (isset(static::$_columns[$name]['class']))
        {
            $objectClass = static::$_columns[$name]['class'];
        }
        else
        {
            $objectClass =
                '\\Orms\\Type\\' .
                    Inflector::classify(static::$_columns[$name]['type']);
        }

        // Si la variable modifiée fait partie des clefs primaires, et que la
        // valeur précédente n'est pas null, on met à jour le champ _pkChanged

        if (in_array($name, static::$_pkFields))
        {
            $this->_pkChanged = true;
        }

        // Mise à jour de la valeur. Si la nouvelle valeur est null, on
        // réinitialise le tout.

        if (! is_null($value))
        {
            // On doit pouvoir affecter un objet à une colonne s'il est de la
            // classe attendue

            if (is_object($value) and ($objectClass === get_class($value)))
            {
                $this->_object_values[$name] = $value;
            }
            else
            {
                $this->_object_values[$name] = new $objectClass($value);
            }

            $this->_string_values[$name] =
                $this->_object_values[$name]->__toString();
        }
        else
        {
            $this->_object_values[$name] = null;
            $this->_string_values[$name] = null;
        }

    }

    /**
     * __get
     *
     * The magic getter
     *
     * @param string $name
     *
     * @return mixed
     */

    public function __get($name)
    {
        // Si la promriété n'existe pas, on lève une exception, en
        // conseillant (sans vraiment donner le choix, certes), d'utiliser
        // external

        if (!isset(static::$_columns[$name]))
        {
            throw new Exception(
                'Property [' . $name . '] does not exist.'
            );
        }

        // Null property

        if (is_null($this->_object_values[$name]))
        {
            if (is_null($this->_string_values[$name]))
            {
                return null;
            }

            // Et donc là on arrive et on met à jour la propriété

            if (isset(static::$_columns[$name]['class']))
            {
                $objectClass = static::$_columns[$name]['class'];
            }
            else
            {
                $objectClass =
                    '\\Orms\\Type\\' .
                        Inflector::classify(static::$_columns[$name]['type']);
            }

            $this->_object_values[$name] =
                new $objectClass($this->_string_values[$name]);
        }

        // Et sinon on renvoie valeur de la propriété

        return $this->_object_values[$name]->getValue();
    }

    /**
     * __isset
     *
     * @param string $name
     *
     * @return mixed
     */

    public function __isset($name)
    {
        return array_key_exists($name, static::$_columns);
    }

    /**
     * find method
     */

    static public function find(
        $provider,
        Array $params = array(),
        Array $links = null)
    {
        $params['countOnly'] = false;

        $result = array();

        foreach ($provider->get($params) as $r)
        {
            $class = get_called_class();

            $result[] = new $class($provider, new \ArrayObject($r), $links);
        }

        return new \ArrayObject($result);
    }

    /**
     * find metadata for the last get method
     */

    static public function findMetaData($provider)
    {
        return $provider->getMetaData();
    }

    /**
     * findNb method
     */

    static public function findNb(
        $provider,
        Array                 $params = array()
    )
    {
        $params['countOnly'] = true;

        return $provider->get($params);
    }

    /**
     * get meta data method
     *
     * @param object $provider
     *
     * @return object
     */

    static public function getMetaData($provider)
    {
        return $provider->getMetaData();
    }

    /**
     * Saving the element to the database. As a matter of fact, this method
     * first try to update the element, and then to insert it. The element is
     * then updated with the data in the database.
     *
     * @return boolean
     *
     * @author Yannick Le Guédart
     */

    public function save()
    {
        if ($this->_savable === false)
        {
            throw new Exception('Object cannot be saved');
        }

        // Exécution de la méthode _beforeSave(), vide par défaut, mais
        // surchargeable dans chaque classe qui le nécessite.

        $this->_beforeSave();

        // Mise à jour des _string_values à partir des _object_values

        $this->_updateStringValues();

        // Si toutes les clefs primaires sont non null, on tente un update,
        // à condition que la valeur _pkChanged soit à false. Si une clef
        // primaire a été modifiée d'une manière quelconque, on ne tentera qu'un
        // insert

        $saveOk = false;

        if (false === $this->_pkChanged)
        {
            $allPkSet = true;

            foreach (static::$_pkFields as $pk)
            {
                $allPkSet = $allPkSet && (!is_null($this->_string_values[$pk]));
            }

            if (true === $allPkSet)
            {
                if ($results =
                        $this->_provider->update(
                            $this->getDefinition(),
                            $this->_string_values
                        )
                )
                {
                    $this->setFrom($results);

                    $saveOk = true;
                }
            }
        }

        // Pas d'update, on fait donc un insert

        if (false === $saveOk)
        {
            if ($results =
                    $this->_provider->insert(
                        $this->getDefinition(),
                        $this->_string_values
                    )
            )
            {
                $this->setFrom($results);

                $saveOk = true;
            }
        }

        if (true === $saveOk)
        {
            $this->_afterSave();
        }

        return $saveOk;
    }

    /**
     * Operations to be done before each save.
     *
     * @return boolean
     *
     * @author Yannick Le Guédart
     */

    protected function _beforeSave()
    {
    }

    /**
     * Operations to be done after each save.
     *
     * @return boolean
     *
     * @author Yannick Le Guédart
     */

    protected function _afterSave()
    {
    }

    /**
     * Delete a row.
     *
     * @return boolean
     *
     * @author Yannick Le Guédart
     */

    public function delete()
    {
        return
            $this->_provider->delete(
                $this->getDefinition(),
                $this->_string_values
            );
    }

    /**
     * set the element from another object or array
     *
     * @param mixed $newElement
     * @param array $except
     *
     * @return boolean
     *
     * @author Yannick Le Guédart
     */

    public function setFrom(
        $newElement,
        $except = array())
    {
        if (!is_object($newElement) and !is_array($newElement))
        {
            throw new Exception(
                'setFrom method first parameter must be array or object, ' .
                gettype($newElement) . ' given.'
            );
        }

        if (!is_array($except))
        {
            throw new Exception(
                'setFrom method second parameter must be array or object, ' .
                gettype($except) . ' given.'
            );
        }

        // Si on a un objet, on le transforme en tableau. Si c'est un objet du
        // type DataObject on récupère les valeurs mises à jour

        if (is_object($newElement))
        {
            if ($newElement instanceof DataObject)
            {
                $newElement = unserialize($newElement->serialize());
            }
            else
            {
                $newElement = get_object_vars($newElement);
            }
        }

        // Si on a un attribut _savage setté, il peut être important d'en tenir
        // Compte

        $savable = true;

        if (isset($newElement['_savable']))
        {
            $this->_savable = ! (false === $newElement['_savable']);
            unset($newElement['_savable']);
        }

        // Puis on boucle. On ignore toutes les entrées ne correspondant pas à
        // une colonne

        foreach ($newElement as $k => $v)
        {
            if (method_exists($this, 'set' . Inflector::classify($k)))
            {
                $method = 'set' . Inflector::classify($k);

                $this->{$method}($v);
            }
            elseif (isset(static::$_columns[$k]) and !in_array($k, $except))
            {
                $this->_string_values[$k] = $v;
                $this->_object_values[$k] = null;
            }
            elseif (!in_array($k, $except))
            {
                $this->external->{$k} = $v;
            }
        }

    }

    /**
     * Update the _string_value array from the _object_value array
     */

    protected function _updateStringValues()
    {
        foreach ($this->_object_values as $k => $v)
        {
            if (is_object($v))
            {
                $this->_string_values[$k] = $v->__toString();
            }
        }
    }

    /**
     * Methode pour sérialiser
     *
     * @return string
     */

    public function serialize()
    {
        $this->_updateStringValues();

        return serialize($this->_string_values);
    }

    /**
     * Methode pour deserialiser
     *
     * @param string $data
     *
     * @return object
     */

    public function unserialize($data)
    {
        $this->setFrom(unserialize($data));

        return $this;
    }

    ////////////////////////////////////////////////////////////////////////////
    // Relation entre éléments - les enfants
    ////////////////////////////////////////////////////////////////////////////

    /**
     * Array containing the table corresponding to the element the current
     * element can have as children
     *
     * @var array
     */

    protected $_linkedElement = array();

    /**
     * Get childs of a given type for an element
     *
     * @param string  $linkedClass
     * @param array   $params
     *
     * @return ArrayObject of $linkedClass object
     *
     * @author Yannick Le Guédart
     */

    public function get(
        $linkedClass = null,
        $params = array()
    )
    {
        $params['countOnly'] = false;

        return $this->_getLinkedElements($linkedClass, $params);
    }

    /**
     * Get one child of a given type for an element.
     *
     * The offset/limit parameters can be provided but are forced to 0/1
     *
     * @param string  $linkedClass
     * @param array   $params
     *
     * @return $linkedClass object or null
     *
     * @author Yannick Le Guédart
     */

    public function getOne(
        $linkedClass = null,
        $params = array()
    )
    {
        $params['countOnly'] = false;
        $params['limit'] = 1;

        $returnArrayObject = $this->_getLinkedElements($linkedClass, $params);

        if ($returnArrayObject->count() > 0)
        {
            return $returnArrayObject->offsetGet(0);
        }

        return null;
    }

    /**
     * Get the first child of a given type for an element.
     *
     * The offset/limit parameters can be provided but are forced to 0/1
     *
     * @param string  $linkedClass
     * @param array   $params
     *
     * @return $linkedClass object or null
     *
     * @author Yannick Le Guédart
     */

    public function getFirst(
        $linkedClass = null,
        $params = array()
    )
    {
        $params['offset'] = 0;

        return $this->getOne($linkedClass, $params);
    }

    /**
     * Get the child metadata
     *
     * @param string  $linkedClass
     *
     * @return object
     *
     * @author Yannick Le Guédart
     */

    public function getChildMetaData($linkedClass = null)
    {
        // ---------------------------------------------------------------------
        // inclusion du namespce nécessaire
        // ---------------------------------------------------------------------

        if (isset($this->_linkedElement[$linkedClass]['targetClass']))
        {
            $targetClass = $this->_linkedElement[$linkedClass]['targetClass'];
        }
        else
        {
            $targetClass = $linkedClass;
        }

        // ---------------------------------------------------------------------
        // We execute the required request an return the result
        // ---------------------------------------------------------------------

        return
            $targetClass::findMetaData(
                $this->_linkedElement[$linkedClass]['provider']
            );
    }

    /**
     * Get number of childs of a given type for an element
     *
     * @param string  $linkedClass
     * @param array   $params
     *
     * @return integer
     *
     * @author Yannick Le Guédart
     */

    public function getNb(
        $linkedClass = null,
        $params = array()
    )
    {
        $params['countOnly'] = true;

        return $this->_getLinkedElements($linkedClass, $params);
    }

    /**
     * Get linked elements or a number of linked elements of a given type for
     * the current element.
     *
     * @param string  $linkedClass
     * @param array   $params
     *
     * @return ArrayObject of $linkedClass object, or integer
     *
     * @author Yannick Le Guédart
     */

    protected function _getLinkedElements(
        $linkedClass = null,
        $params = array()
    )
    {
        // ---------------------------------------------------------------------
        // At first, we check if the required elements are indeed contained
        // in the current element
        // ---------------------------------------------------------------------

        $this->_checkLinkedElementType($linkedClass);

        // ---------------------------------------------------------------------
        // inclusion du namespce nécessaire
        // ---------------------------------------------------------------------

        if (isset($this->_linkedElement[$linkedClass]['targetClass']))
        {
            $targetClass = $this->_linkedElement[$linkedClass]['targetClass'];
        }
        else
        {
            $targetClass = $linkedClass;
        }

        // ---------------------------------------------------------------------
        // Link the current class to the linked element
        // ---------------------------------------------------------------------

        foreach ($this->_linkedElement[$linkedClass]['params'] as $k => $v)
        {
            if (preg_match('/^\{(.*?)\}$/', $v, $m))
            {
                if (strpos($m[1], ':') !== false)
                {
                    $p = $this;

                    foreach (split(':', $m[1]) as $attr)
                    {
                        $p = $p->{$attr};
                    }

                    $params[$k] = $p;
                }
                elseif (!is_null($this->{$m[1]})
                )
                {
                    $params[$k] = $this->{$m[1]};
                }
                else
                {
                    throw new Exception(
                        'Null parameter ' . $m[1] . '/' . $k
                    );
                }
            }
            else
            {
                $params[$k] = $v;
            }

        }

        // ---------------------------------------------------------------------
        // We execute the required request an return the result
        // ---------------------------------------------------------------------

        if ($params['countOnly'] === true)
        {
            return
                    $targetClass::findNb(
                        $this->_linkedElement[$linkedClass]['provider'],
                        $params
                    );
        }
        else
        {
            return
                    $targetClass::find(
                        $this->_linkedElement[$linkedClass]['provider'],
                        $params,
                        $this->_linkedElement[$linkedClass]['links']
                    );
        }
    }

    ////////////////////////////////////////////////////////////////////////////
    // Checkers
    ////////////////////////////////////////////////////////////////////////////

    /**
     * Checks if the current object can have linked elements of the given class.
     * throws an exception if not.
     *
     * @param string $linkedClass
     *
     * @return void
     *
     * @throws CoreException
     *
     * @author Yannick Le Guédart
     */

    protected function _checkLinkedElementType($linkedClass)
    {
        if (!isset($this->_linkedElement[$linkedClass]))
        {
            throw new Exception(
                "[$linkedClass] Children not defined."
            );
        }
    }
}
