<?php

namespace Orms\Proxy\Dumper;

use Orms\Proxy\MethodDumper;

class ProxyDumper
{
    const CLASS_TEMPLATE = <<<EOF
namespace Proxies;

class %s extends \\%s
{
%s
}\n
EOF;

    /**
     * @var MethodDumperInterface
     */
    private $defaultDumper;

    /**
     * @var MethodDumperInterface[]
     */
    private $dumpers;

    /**
     * @param MethodDumperInterface[] $dumpers
     * @param MethodDumperInterface   $defaultDumper
     */
    public function __construct(array $dumpers = [], MethodDumperInterface $defaultDumper = null)
    {
        $this->defaultDumper = $defaultDumper ?: new IsoMethodDumper();
        $this->dumpers       = $dumpers;
    }

    /**
     * @param  string                $name
     * @param  MethodDumperInterface $dumper
     *
     * @return ProxyDumper
     */
    public function registerDumper($name, MethodDumperInterface $dumper)
    {
        $this->dumpers[$name] = $dumper;

        return $this;
    }

    /**
     * @param MethodDumperInterface[] $dumpers
     */
    public function setDumpers(array $dumpers)
    {
        $this->dumpers = $dumpers;
    }

    /**
     * @param \ReflectionClass $class
     * @param string           $configName The entity configuration name in orms.
     *
     * @return string
     */
    public function dump(\ReflectionClass $class, $configName)
    {
        return sprintf(
            self::CLASS_TEMPLATE,
            $class->getShortName(),
            $class->getName(),
            $this->dumpMethods($class, $this->getOverridableMethods($class), $configName)
        );
    }

    /**
     * @param \ReflectionClass    $class
     * @param \ReflectionMethod[] $methods
     * @param string              $configName The entity configuration name in orms.
     *
     * @return string
     */
    private function dumpMethods(\ReflectionClass $class, array $methods, $configName)
    {
        $string = [];
        foreach ($methods as $method) {
            $key = sprintf('%s::%s', $class->name, $method->name);
            $string[]= array_key_exists($key, $this->dumpers)
                ? $this->dumpers[$key]->dump($method, $configName)
                : $this->defaultDumper->dump($method, $configName);
        }

        return implode("\n\n", array_filter($string, function ($method) {
            return !empty($method);
        }));
    }

    /**
     * @param \ReflectionClass $class
     *
     * @return \ReflectionMethod[]
     */
    private function getOverridableMethods(\ReflectionClass $class)
    {
        return array_filter($class->getMethods(\ReflectionMethod::IS_PUBLIC | \ReflectionMethod::IS_PROTECTED), function ($method) {
            return !$method->isStatic() && !$method->isFinal() && !$method->isAbstract();
        });
    }
}
