<?php

namespace Orms\DataProvider;

use Orms\Exception\Exception;
use Orms\DataProvider\GenericDataProvider;
use Orms\Statement\SolrStatement;
use Orms\Exception\SolrQueryException;

abstract class SolrDataProvider extends GenericDataProvider
{
    public function setup($options)
    {
    }

    /**
     * Inserts some data.
     *
     * @param array $definition
     * @param array $values
     *
     * @return mixed - returns Array of $values after the insertion || throw an
     * exception
     *
     * @author Yannick Le Guédart
     */

    public function insert(
        Array $definition,
        Array $values
    )
    {
        // insertion ?
    }

    /**
     * Update a row.
     *
     * @param array $definition
     * @param array $values
     *
     * @return mixed - returns Array of $values after the insertion || throw an
     * exception
     *
     * @author Yannick Le Guédart
     */

    public function update(
        Array $definition,
        Array $values
    )
    {
        // Update ?
    }

    /**
     * Delete a row.
     *
     * @param array $definition
     * @param array $values
     *
     * @return boolean
     *
     * @author Yannick Le Guédart
     */

    public function delete(
        Array $definition,
        Array $values
    )
    {
        // Delete
    }

    public function get(Array $params)
    {
        $params['_connection'] = $this->getConnection();

        return parent::get($params);
    }

    /**
     * Paramètres de base d'une méthode get
     *
     * Ce tableau fixe les paramètres de base d'une requête get à moins d'une
     * réécriture dans une classe fille. Il n'est pas nécessaire de réécrire
     * tous les paramètres.
     *
     * @see get()
     *
     * @var    array
     */

    static public $_getBaseParams =
        array
        (
            'countOnly'          => false,

            'offset'             => 0,
            'limit'              => 100,

            'sortFields'         => array(), // field => ASC/DESC / Deprcated
            'order'              => array(), // Same as sortfield

            'dummy_param'        => array(), // array (key => value)

            'search'             => null,
            'filter'             => array(), // array (fields => search string)
            'facet'              => null,    // array (facet_name => array(params))

            'facet_range'        => null,    // array (fields => array(params))

            '_onlyFields'        => null,    // array of fields
            '_connection'        => null,
        );

    static public $_getFilterParams =
        array
        (
            'id'       => null,
        );

    /**
     * get composer
     */

    static protected function getComposer($params)
    {
        if (!is_array($params))
        {
            throw new Exception('$params is not an array');
        }

        // Tableaux de paramètres par défaut.

        $baseDefault = array();
        $filterDefault = array();

        // Tableaux de paramètres réels.

        $filterParams = array();

        $classLoop = get_called_class();

        while ($classLoop)
        {
            foreach ($classLoop::$_getBaseParams as $k => $v)
            {
                if (!isset($baseDefault[$k]))
                {
                    $baseDefault[$k] = $v;
                }
            }

            foreach ($classLoop::$_getFilterParams as $k => $v)
            {
                if (!isset($filterDefault[$k]))
                {
                    $filterDefault[$k] = $classLoop;

                    if (!is_null($v))
                    {
                        $filterParams[$k] = $v;
                    }
                }
            }

            if ($classLoop === 'GenericDataProvider')
            {
                break;
            }

            $classLoop = get_parent_class($classLoop);
        }

        /* ---------------------------------------------------------------------
           * Génération des tableaux de paramètres réels
           * ---------------------------------------------------------------------
           */

        $baseParams = $baseDefault;

        foreach ($params as $k => $v)
        {
            if (array_key_exists($k, $baseParams))
            {
                $baseParams[$k] = $v;
            }
            elseif (array_key_exists($k, $filterDefault))
            {
                $filterParams[$k] = $v;
            }
            else
            {
                throw new \Orms\Exception\Exception(
                    'Invalid query parameter [' . $k . ']'
                );
            }
        }

        /* ---------------------------------------------------------------------
           * Génération du tableau de bouts de requetes SQL utilisées poru les
           * filtres.
           * ---------------------------------------------------------------------
           */

        $client = $baseParams['_connection'];
        $query = new \SolrQuery();

        foreach ($filterParams as $k => $v)
        {
            if (! is_null($v) && ! empty($v))
            {
                $query->addFilterQuery($k . ':' . $v);
            }
        }

        /* ---------------------------------------------------------------------
         * Gestion des paramètres de base
         * ---------------------------------------------------------------- */

        if (true === $baseParams['countOnly'])
        {
            $baseParams['offset'] = 0;
            $baseParams['limit']  = 0;
        }

        if (is_int($baseParams['offset']) && $baseParams['offset'] >= 0)
        {
            $query->setStart($baseParams['offset']);
        }

        if (is_int($baseParams['limit']) && $baseParams['limit'] >= 0)
        {
            $query->setRows($baseParams['limit']);
        }

        /**
         * Gestion du paramètre de recherche
         *
         * - Premier cas, le paramètre est null. Dans ce cas, on recherche tout
         *   et n'importe quoi (*:*)
         * - Second cas. On n'a qu'une chaine => on la transforme en un tableau,
         *   histoire de tout gérer de la même manière.
         * - Cas général : on a un tableau de clefs +> valeurs. Chacune de ces
         *   ensembles est un paramètre de recherche. C'est un ensemble de ||,
         *   SolR affectant un rank en fonction des résultats. Pour chaque
         *   valeur, on escape la chaine, SAUF les * qui constituent les jokers
         *   lors de recherches exactes.
         */

        if (is_null($baseParams['search']))
        {
            $query->setQuery('*:*');
        }
        else
        {
            if (! is_array($baseParams['search']))
            {
                $baseParams['search'] = [$baseParams['search']];
            }

            $q = [];

            foreach ($baseParams['search'] as $k => $v)
            {
                /**
                 * if the search value is an array, then we have a text && a
                 * weight for this search
                 */

                if (is_array($v))
                {
                    $weight = $v[1];
                    $v      = $v[0];
                }
                else
                {
                    $weight = null;
                }

                $chunks = [];

                foreach (explode('*', $v) as $chunk)
                {
                    $chunks[] = \SolrUtils::escapeQueryChars($chunk);
                }

                $q[] =
                    '(' . $k . ':(' . implode('*', $chunks) . ')' .
                    (is_null($weight) ? '' : "^$weight") . ')';
            }

            $query->setQuery('(' . implode(' || ', $q) . ')');
        }

        foreach ($baseParams['filter'] as $k => $v)
        {
            if (! is_null($v) && ! empty($v))
            {
                if (is_array($v))
                {
                    $query->addFilterQuery(
                        $k .
                        ':(' . implode(' ', $v) . ')'
                    );
                }
                else
                {
                    $query->addFilterQuery($k . ':' . $v);
                }
            }
        }

        /** --------------------------------------------------------------------
         *  Handling facets && facet ranges
         *  ----------------------------------------------------------------- */

        // Facets

        if (
            isset($baseParams['facet'])
            &&
            is_array($baseParams['facet'])
            &&
            (count($baseParams['facet']) > 0)
        )
        {
            $query->setFacet(true);

            foreach ($baseParams['facet'] as $f => $p)
            {
                $query->addFacetField($f);

                foreach ($p as $k => $v)
                {
                    $query->addParam('f.' . $f . '.facet.' . $k, $v);
                }
            }
        }

        // Range

        if (
            isset($baseParams['facet_range'])
            &&
            is_array($baseParams['facet_range'])
            &&
            (count($baseParams['facet_range']) > 0)
        )
        {
            $query->setFacet(true);

            foreach ($baseParams['facet_range'] as $f => $p)
            {
                $query->addParam('facet.range', $f);

                foreach ($p as $k => $v)
                {
                    $query->addParam('f.' . $f . '.facet.range.' . $k, $v);
                }
            }
        }

        /**
         * dummy params
         */

        foreach ($baseParams['dummy_param'] as $k => $v)
        {
            $query->addParam($k, $v);
        }

        /**
         * Handling sort fields
         */

        foreach ($baseParams['order'] as $k => $v)
        {
            $query->addSortField($k, $v);
        }

        foreach ($baseParams['sortFields'] as $k => $v)
        {
            $query->addSortField($k, $v);
        }

        if (! is_null($baseParams['_onlyFields']))
        {
            foreach ($baseParams['_onlyFields'] as $f)
            {
                $query->addField($f);
            }
        }

        /* ---------------------------------------------------------------------
         * Exécution de la requête SolR.
         *
         * En fonction du type de requète, $baseParams['countOnly'] ou pas, on
         * revoit un entier ou un ArrayObject.
         * ---------------------------------------------------------------------
         */

        $logger = self::getLogger();
        self::$_meta[get_called_class()] = new \StdClass();

        if ($logger)
        {
            $logParam = $params;

            unset($logParam['_connection']);

            $logger->startQuery($query->__toString(), $logParam);
        }

        try
        {
            $query_response = $client->query($query);
        }
        catch(\Exception $e)
        {
            $e = new SolrQueryException($e);

            if ($logger)
            {
                $logger->stopQuery();
                $logger->logException($e->getErrorInfo());
            }

            throw $e;
        }

        $rs = $query_response->getResponse();

        if ($logger)
        {
            $logger->stopQuery();
        }

        /**
         * Saving facets && ranges
         */

        if (true === $query->getFacet())
        {
            if (
                isset($baseParams['facet'])
                &&
                is_array($baseParams['facet'])
                &&
                (count($baseParams['facet']) > 0)
            )
            {
                self::$_meta[get_called_class()]->facet = new \stdClass();

                foreach ($rs->facet_counts->facet_fields as $k => $v)
                {
                    self::$_meta[get_called_class()]->facet->{$k} = (array) $v;
                }
            }

            if (
                isset($baseParams['facet_range'])
                &&
                is_array($baseParams['facet_range'])
                &&
                (count($baseParams['facet_range']) > 0)
            )
            {
                self::$_meta[get_called_class()]->facet_range = new \stdClass();

                foreach ($rs->facet_counts->facet_ranges as $k => $v)
                {
                    self::$_meta[get_called_class()]->facet_range->{$k} =
                        array();

                    foreach ($v->counts as $sk => $sv)
                    {
                        self::$_meta[
                        get_called_class()
                        ]->facet_range->{$k}[$sk] = $sv;
                    }
                }
            }
        }

        /**
         * Getting results
         */

        self::$_meta[get_called_class()]->numFound =
            intval($rs->response->numFound);

        if ($baseParams['countOnly'])
        {
            return intval($rs->response->numFound);
        }
        else
        {
            return new SolrStatement($rs->response);
        }
    }
}
