<?php

namespace Orms\DataProvider;

use Orms\Exception\Exception;
use Orms\Exception\QueryException;

use Orms\DataProvider\GenericDataProvider;

use Orms\Statement\MssqlStatement;
use Orms\Statement\Statement;

abstract class MssqlDataProvider extends GenericDataProvider
{
    /**
     * shema & table name
     */

    protected $schema;
    protected $table;

    /**
     * @param mixed $options
     */

    public function setup($options)
    {
        if (! array_key_exists('schema', $options))
        {
            throw new Exception(
                "MssqlDataProvider : \$options['schema'] is missing."
            );
        }

        if (! array_key_exists('table', $options))
        {
            throw new Exception(
                "MssqlDataProvider : \$options['table'] is missing."
            );
        }

        $this->schema = $options['schema'];
        $this->table  = $options['table'];
    }

    /**
     * Inserts some data.
     *
     * @param array $definition
     * @param array $values
     *
     * @return mixed - returns Array of $values after the insertion
     *
     * @throws QueryException
     *
     * @author Yannick Le Guédart
     */

    public function insert(
        Array $definition,
        Array $values
    )
    {
        // génération des noms de colonnes

        $columns      = [];
        $quotedValues = [];

        $i = 1;

        foreach ($definition['columns'] as $name => $def)
        {
            // Si c'est une clef primaire et qu'elle est à null, on l'ignore,
            // ainsi que si elle est à null et qu'elle est indiqué comme ayant
            // une valeur par défaut

            if (
                is_null($values[$name])
                &&
                (
                    in_array($name, $definition['pkFields'])
                    ||
                    (
                            isset($def['has_default'])
                            &&
                            ($def['has_default'] === true)
                    )
                )
            )
            {
                continue;
            }

            // gestion des booléens

            if ($def['type'] == 'boolean' && is_bool($values[$name]))
            {
                $values[$name] = ($values[$name] === true) ? '1' : '0';
            }

            // préparation des données

            $quotedValues[] = $this->getConnection()->quote($values[$name]);
        }

        // Génération de la requète d'insertion

        $sql =
            trim(
                'INSERT INTO ' . $this->_getTableName() .
                    '(' . implode(',', $columns) . ') ' .
                'VALUES ' .
                    '(' . implode(',', $quotedValues) . ');'
            );

        // Execution

        $logger = self::getLogger();

        self::$_meta[get_called_class()] = new \StdClass();

        if ($logger)
        {
            $logger->startQuery($sql);
        }

        if ($this->getConnection()->exec($sql))
        {
            if ($logger)
            {
                $logger->stopQuery();
            }

            self::$_meta[get_called_class()]->rowCount = 1;

            // Ca a marché, on récupère la ligne enregistrée, ce qui en mssql
            // n'est possible à priori que si on n'a qu'une seule clef primaire

            if (count($definition['pkFields']) !== 1)
            {
                return false;
            }

            if (isset($values[$definition['pkFields'][0]]))
            {
                return
                    $this
                        ->get(
                            [
                                $definition['pkFields'][0] =>
                                    $values[$definition['pkFields'][0]]
                            ]
                        )
                        ->current();
            }
            else // We have to look to the last inserted ID
            {
                $id =
                    $this
                        ->getConnection()
                        ->query('SELECT SCOPE_IDENTITY()')
                        ->fetch(MssqlStatement::FETCH_NUM);

                return
                    $this
                        ->get([ $definition['pkFields'][0] => $id[0] ])
                        ->current();
            }
        }
        else
        {
            $e = new QueryException();

            if ($logger)
            {
                $logger->logException($e->getErrorInfo());
            }

            self::$_meta[get_called_class()]->errorCode = $sth->errorCode();
            self::$_meta[get_called_class()]->errorInfo = $sth->errorInfo();

            throw $e;
        }
    }

    /**
     * Update a row.
     *
     * @param array $definition
     * @param array $values
     *
     * @return mixed - returns Array of $values after the insertion || throw an
     * exception
     *
     * @author Yannick Le Guédart
     */

    public function update(
        Array $definition,
        Array $values
    )
    {
        // verifying that primary keys are set

        $pks = [];

        foreach ($definition['pkFields'] as $pk)
        {
            if (is_null($values[$pk]))
            {
                throw new Exception("Primary key $pk not set on update");
            }

            $pks[] = $pk . ' = ' . $this->getConnection()->quote($values[$pk]);
        }

        // génération des noms de colonnes

        $columns = [];

        foreach ($definition['columns'] as $name => $def)
        {
            // gestion des booléens

            if ($def['type'] == 'boolean' && is_bool($values[$name]))
            {
                $values[$name] = ($values[$name] === true) ? '1' : '0';
            }

            // préparation des données

            $columns[] =
                $name . ' = ' . $this->getConnection()->quote($values[$name]);
        }

        // Génération de la requète d'update

        $sql =
            trim(
                'UPDATE ' . $this->_getTableName() . ' SET ' .
                    implode(',', columns) .
                ' WHERE ' . implode(' AND ', $pks) . ';'
            );

        // Execution

        $logger                          = self::getLogger();
        self::$_meta[get_called_class()] = new \StdClass();

        if ($logger)
        {
            $logger->startQuery($sql);
        }

        if ($this->getConnection()->exec($sql))
        {
            if ($logger)
            {
                $logger->stopQuery();
            }

            /** On récupère l'objet mis à jour à partir de ses clefs
             * primaires */

            $params = [];

            foreach ($definition['pkFields'] as $pk)
            {
                $params[$pk] = $values[$pk];
            }

            $result = $this->get($params);

            if ($result->count() === 0)
            {
                return false;
            }

            self::$_meta[get_called_class()]->rowCount = 1;

            return $result->current();
        }
        else
        {
            $e = new QueryException();

            if ($logger)
            {
                $logger->logException($e->getErrorInfo());
            }

            self::$_meta[get_called_class()]->errorCode = $sth->errorCode();
            self::$_meta[get_called_class()]->errorInfo = $sth->errorInfo();

            throw $e;
        }
    }

    /**
     * Delete a row.
     *
     * @param array $definition
     * @param array $values
     *
     * @return boolean
     *
     * @author Yannick Le Guédart
     */

    public function delete(
        Array $definition,
        Array $values
    )
    {
        // verifying that primary keys are set

        $pks = [];

        foreach ($definition['pkFields'] as $pk)
        {
            if (is_null($values[$pk]))
            {
                throw new Exception("Primary key $pk not set on update");
            }

            $pks[] = $pk . ' = ' . $this->getConnection()->quote($values[$pk]);
        }

        // Génération de la requète d'insertion

        $sql =
            trim(
                'DELETE FROM ' . $this->_getTableName() . ' ' .
                    'WHERE ' . implode(' AND ', $pks)
            );

        // Execution

        $logger      = self::getLogger();
        self::$_meta[get_called_class()] = new \StdClass();

        if ($logger)
        {
            $logger->startQuery($sql, $values);
        }

        $result = $this->getConnection()->exec($sql);

        self::$_meta[get_called_class()]->rowCount = $result;

        if ($logger)
        {
            $logger->stopQuery();
        }

        return (bool) $result;
    }


    public function get(Array $params)
    {
        $params['tableName'] = $this->_getTableName();
        $params['_connection'] = $this->getConnection();

        return parent::get($params);
    }

    /**
     * Returns the full tablename in the postgresqldatabase
     *
     * @return string
     */

    protected function _getTableName()
    {
        return
            (
                (
                    is_null($this->schema)
                        ?
                        ''
                        :
                        $this->schema . '.'
                )
                . $this->table
            );
    }

    /**
     * get composer
     */

    static public $_getBaseParams =
        [
            'tableName'   => null,
            '_connection' => null
        ];

    static public $_getFilterParams =
        [
            'id'       => null,
            'id_array' => null
        ];

    static protected function getComposer($params)
    {
        if (! is_array($params))
        {
            throw new Exception('$params is not an array');
        }

        // Tableaux de paramètres par défaut.

        $baseDefault   = [];
        $filterDefault = [];

        // Tableaux de paramètres réels.

        $filterParams = array();

        $classLoop = get_called_class();

        while ($classLoop)
        {
            foreach ($classLoop::$_getBaseParams as $k => $v)
            {
                if (!isset($baseDefault[$k]))
                {
                    $baseDefault[$k] = $v;
                }
            }

            foreach ($classLoop::$_getFilterParams as $k => $v)
            {
                if (!isset($filterDefault[$k]))
                {
                    $filterDefault[$k] = $classLoop;

                    if (!is_null($v))
                    {
                        $filterParams[$k] = $v;
                    }
                }
            }

            if ($classLoop === 'GenericDataProvider')
            {
                break;
            }

            $classLoop = get_parent_class($classLoop);
        }

        /* ---------------------------------------------------------------------
           * Génération des tableaux de paramètres réels
           * ---------------------------------------------------------------------
           */

        $baseParams = $baseDefault;

        foreach ($params as $k => $v)
        {
            if (array_key_exists($k, $baseParams))
            {
                $baseParams[$k] = $v;
            }
            elseif (array_key_exists($k, $filterDefault))
            {
                $filterParams[$k] = $v;
            }
            else
            {
                throw new \Orms\Exception\Exception(
                    'Invalid query parameter [' . $k . ']'
                );
            }
        }

        /* ---------------------------------------------------------------------
           * Génération du tableau de bouts de requetes SQL utilisées poru les
           * filtres.
           * ---------------------------------------------------------------------
           */

        $t  = $baseParams['tableName'];
        $db = $baseParams['_connection'];

        $filterSQLArray = [];

        foreach ($filterParams as $k => $v)
        {
            if (!is_null($v))
            {
                $oneFilter =
                    call_user_func_array
                    (
                        array
                        (
                            $filterDefault[$k],
                            '_getFilterSQL'
                        ),
                        array($k, $v, $t, $db)
                    );

                if (!is_null($oneFilter) && ($oneFilter !== ''))
                {
                    $filterSQLArray[] = $oneFilter;
                }
            }
        }

        if (count($filterSQLArray) > 0)
        {
            $filterSQL =
                "WHERE " . implode(' AND ', array_unique($filterSQLArray));
        }
        else
        {
            $filterSQL = '';
        }

        /* ---------------------------------------------------------------------
         * Gestion des paramètres par défaut.
         *
         * C'est ici qu'on va gérer les options par défaut.
         *
         * Tout d'abord, $baseParams['countOnly'] détermine si l'ont fait un
         * décompte ou si l'on veut des éléments. Dans le cas d'un décompte, on
         * peut ignorer tous les paramètres du tableau de base, mais aussi tous
         * paramètres de champs supplémentaires.
         *
         * On commence à construire la chaine des champs à recupérer dans la
         * requète, la chaine liée à l'order by, ainsi que les offset et limit.
         * ---------------------------------------------------------------------
         */

        $orderSQL  = null;
        $offsetSQL = '';
        $limitSQL  = '';
        $_savable  = true;

        if ($baseParams['countOnly'] === true)
        {
            $fieldSQL = array("COUNT ($t.*) AS c");
        }
        else
        {
            /* -----------------------------------------------------------------
             * Offset / Limit
             *
             * Un offset de 0 ou un offset non numérique est ignoré.
             * -----------------------------------------------------------------
             */

            if (is_numeric($baseParams['offset'])
                && ($baseParams['offset'] > 0)
            )
            {
                // Something awful to do (something to do with
            }

            if (! is_null($baseParams['limit']))
            {
                $limitSQL = 'TOP ' . $baseParams['limit'];
            }

            /** Handling order */

            if (! is_null($baseParams['order']))
            {
                if (! is_array($baseParams['order']))
                {
                    throw new \Orms\Exception\Exception(
                        'Invalid order parameter.'
                    );
                }
                else
                {
                    $orderSQLArray = [];

                    foreach ($baseParams['order'] as $field => $asc)
                    {
                        $orderSQLArray[] =
                            "$t.$field " . ((true === $asc) ? 'ASC' : 'DESC');
                    }

                    $orderSQL = 'ORDER BY ' . implode(', ', $orderSQLArray);
                }
            }
            elseif (! is_null($baseParams['orderType']))
            {
                $orderSQL =
                    'ORDER BY ' . $baseParams['orderType'] . ' ' .
                        (($baseParams['orderAsc'] === true) ? 'ASC' : 'DESC');
            }

            if (
                is_array($baseParams['_onlyFields'])
                &&
                (count($baseParams['_onlyFields']) > 0)
            )
            {
                $fieldSQL = array();

                foreach ($baseParams['_onlyFields'] as $field)
                {
                    $fieldSQL[] = "$t.$field";
                }

                $_savable = false;
            }
            else
            {
                $fieldSQL = array("$t.*");
            }
        }

        /**
         * On injecte la variable _savable dans la requète
         * pour l'avoir directement dans l'objet
         */

        $fieldSQL[] = sprintf('%s AS _savable', (int) $_savable);

        /* ---------------------------------------------------------------------
         * Création de la requête SQL.
         * ---------------------------------------------------------------------
         */

        $sql =
            trim(
                "SELECT " .
                    $limitSQL .
                    join(',', $fieldSQL) .
                " FROM $t " .
                "$filterSQL " .
                "$orderSQL " .
                "$limitSQL"
            );

        /* ---------------------------------------------------------------------
         * Exécution de la requête SQL.
         *
         * En fonction du type de requète, $baseParams['countOnly'] ou pas, on
         * revoit un entier ou un ArrayObject.
         * ---------------------------------------------------------------------
         */

        $logger = self::getLogger();

        if ($logger)
        {
            $logParam = $params;

            unset($logParam['_connection'], $logParam['tableName']);

            $logger->startQuery($sql, $logParam);
        }

        $rs = $db->query($sql);

        self::$_meta[get_called_class()] = new \StdClass();

        if ($logger)
        {
            $logger->stopQuery();
        }

        // If PDO object is empty, then query id corrupt

        if (false === $rs)
        {
            $e = new QueryException($db);

            if ($logger)
            {
                $logger->logException($e->getErrorInfo());
            }

            self::$_meta[get_called_class()]->errorCode = $db->errorCode();
            self::$_meta[get_called_class()]->errorInfo = $db->errorInfo();

            throw $e;
        }

        if ($baseParams['countOnly'])
        {
            $record = $rs->fetch(Statement::FETCH_ASSOC);

            self::$_meta[get_called_class()]->rowCount = 1;

            return intval($record['c']);
        }
        else
        {
            // Save result count in meta
            self::$_meta[get_called_class()]->rowCount = $rs->rowCount();

            return $rs;
        }
    }

}
