<?php

namespace Orms\DataProvider;

use Orms\Object\DataObject;
use Orms\Logger\LoggerInterface;
use Orms\EventDispatcher\EventDispatcherInterface;
use Orms\Middleware\QueryCacheInterface;
use Orms\Middleware\CacheResult;

abstract class GenericDataProvider
{
    const ORDER_ASC  = true;
    const ORDER_DESC = false;

    /**
     * Connection - can be any data provider from a database to a csv
     * file || a rest service
     */

    protected $_connection;

    /**
     * Store an external context to be accessible in model
     *
     * @var object
     */

    protected static $externalContextInstance;

    /**
     * Logger instance
     *
     * @var LoggerInterface
     */

    protected static $_logger;

    /**
     * Dispatcher instance
     * @var EventDispatcherInterface
     */

    protected $_dispatcher;

    /**
     * Cache Middleware instance
     * @var QueryCacheInterface
     */
    protected $_cache;

    /**
     * Metadata object
     *
     * @var array of meta objects
     */

    protected static $_meta = [];

    /**
     * Constructor - create the connection to the service used.
     *
     * @param Connection primary keys of the data provided
     * @param mixed $options
     * @param $externalContext
     * @param LoggerInterface $logger
     * @param EventDispatcherInterface $dispatcher
     * @param QueryCacheInterface $cache
     */

    public function __construct(
        $connection,
        $options = null,
        $externalContext = null,
        LoggerInterface $logger = null,
        EventDispatcherInterface $dispatcher = null,
        QueryCacheInterface $cache = null
    ) {
        $this->_connection               = $connection;
        static::$externalContextInstance = $externalContext;
        static::$_logger                 = $logger;
        $this->_dispatcher               = $dispatcher;
        $this->_cache                    = $cache;

        $this->setup($options);
    }

    /**
     * Return connection Handler
     * @return array
     */

    public function getConnection()
    {
        return $this->_connection->getConnection();
    }

    /**
     * Return external context
     * @return mixed
     */

    public static function getExternalContext()
    {
        return static::$externalContextInstance;
    }

    /**
     * Remove connection reference
     */
    public function __destruct()
    {
        unset($this->_connection);
    }

    /**
     * @param mixed $options
     */

    abstract public function setup($options);

    /**
     * @param array $definition
     * @param array $values
     *
     * @return array (value of the object after the insert)
     */

    abstract public function insert(
        array $definition,
        array $values
    );

    /*
      * @param array $definition
      * @param array $values
      *
      * @return array (value of the object after the insert)
      */

    abstract public function update(
        array $definition,
        array $values
    );

    /*
      * @param array $definition
      * @param array $values
      *
      * @return boolean (wether the delete was successful || not)
      */

    abstract public function delete(
        array $definition,
        array $values
    );

    /**
     * getter / Query composer
     *
     * @param array $definition
     * @param array $values
     *
     * Toute la composition des query composer est statique car on se sert de
     * l'héritage pour éviter de répéter des données dans les tableaux de
     * config.
     */

    public function get(array $params)
    {
        // Read Cache
        if ($result = $this->getFromCache($params)) {
            return $result;
        }

        // Exec Query
        $result = static::getComposer($params);

        // Save to cache
        $this->saveToCache($result, $params);

        return $result;
    }

    /**
     * Read data from Cache Middleware
     * @param array $params
     * @return mixed
     */
    private function getFromCache(array $params)
    {
        if (is_null($this->_cache)) {
            return false;
        }

        try {
            if ($result = $this->_cache->handleGet($params)) {
                // Set metadata
                static::$_meta[static::class] = $result->getMetadata();

                // Log cached query
                static::getLogger()->startQuery($result->getQuery(), $params);
                static::getLogger()->stopQuery(true, count($result->getResult()));

                return $result->getResult();
            }
        } catch (\Exception $e) {
            static::getLogger()->logException($e->getMessage());
        }

        return false;
    }

    /**
     * Save data to Cache Middleware
     * @param mixed $result
     * @param array $params
     */
    private function saveToCache($result, array $params)
    {
        if (is_null($this->_cache)) {
            return;
        }

        try {
            $this->_cache->handleSave($params, new CacheResult(
                    $result,
                    $this->getMetaData(),
                    static::getLogger()->getLastExecutedQuery()
                ));
        } catch (\Exception $e) {
            static::getLogger()->logException($e->getMessage());
        }
    }

    /**
     * get last execution meta data
     *
     * @return object
     */

    public function getMetaData()
    {
        return static::$_meta[static::class];
    }

    /**
     * Paramètres de base d'une méthode get
     *
     * Ce tableau fixe les paramètres de base d'une requête get à moins d'une
     * réécriture dans une classe fille. Il n'est pas nécessaire de réécrire
     * tous les paramètres.
     *
     * @see get()
     *
     * @deprecated 2.2.0 use the getBaseParams method instead
     *
     * @var    array
     */

    public static $_getBaseParams =

        [
            'countOnly'   => false,
            'offset'      => 0,
            'limit'       => null,
            'orderType'   => null,
            'orderAsc'    => true,

            // New way of handling orders

            'order'       => null,

            '_onlyFields' => null,
            '_countBy'    => null,
            '_groupBy'    => null,
        ];

    /**
     * Paramètres pour filtrer les données d'une méthode get
     *
     * Ce tableau définit comment les données seront filtrées d'une requete get.
     * parent_id ou parent_path sont généralement settés par le paramettre
     * $parentParam de la méthode get, même si un filtrage est toujours possible
     * en sus. Il est tout à fait possible qu'un filtre nécessite une jointure.
     *
     * Si l'on veut qu'une requète get soit filtrée en permanence, il faut
     * définir le filtre par défaut avec une valeur non nulle.
     *
     * @deprecated 2.2.0 use the getFilterParams method instead
     *
     * @var array
     */

    public static $_getFilterParams = [];

    /**
     * La grande méthode du get, appelée par les méthodes de la classe
     * jDbObject et toute sa déscendance. Cette méthode utilise les
     * tableaux de configuration définis ci-dessus
     *
     * @param array   $params
     *
     * @return integer (count) || arrayObject
     *
     * @author Yannick Le Guédart
     */

    protected static function getComposer($params)
    {
        return;
    }

    /**
     * Return the logger instance
     *
     * @return LoggerInterface
     */

    protected static function getLogger()
    {
        return static::$_logger;
    }

    /**
     * Notify Event Dispatcher of a DataObject write
     *
     * @param DataObject $object
     * @param int $eventType
     */

    public function notify(DataObject $object, $eventType)
    {
        if (!is_null($this->_dispatcher)) {
            $this->_dispatcher->notify($object, $eventType);
        }
    }

    /**
     * @return array
     */
    protected static function computeBaseParams()
    {
        $class = static::class;

        return method_exists($class, 'getBaseParams') ?
            $class::getBaseParams() : $class::$_getBaseParams;
    }

    /**
     * @return array
     */
    protected static function computeFilterParams()
    {
        $class = static::class;

        return method_exists(static::class, 'getFilterParams') ?
            $class::getFilterParams() : $class::$_getFilterParams;
    }
}
