<?php

namespace Orms\Type;

/**
 * Class for point type in postgresql (could be used in any other SGBD using
 * a string)
 */

class Point extends Type implements TypeInterface
{
    /**
     * X coordinate
     *
     * @access protected
     * @var float
     */

    protected $_x = null;

    /**
     * Y coordinate
     *
     * @access protected
     * @var float
     */

    protected $_y = null;

    /**
     * sets the value of the object after checking
     *
     * @param mixed $value
     *
     * @return object
     */

    public function setValue($value)
    {
        parent::setValue($value);

        $this->_decode($value);
    }

    /**
     * Gets an usable value of the object
     *
     * @return mixed
     */

    public function getValue()
    {
        return $this;
    }

    /**
     * Checks if the value provided is valid
     *
     * @param mixed $value
     *
     * @returns boolean
     */

    public function isValid($value)
    {
        // $value must be a string || an array

        if (is_array($value))
        {
            return (
                2 === count($value)
                &&
                is_numeric($value[0])
                &&
                is_numeric($value[1])
            );
        }
        elseif (is_string($value))
        {
            // validation de la chaine. On considère pour l'instant que la
            // chaine du store est valide si elle a au moins une clef/valeur

            $valid =
                preg_match(
                    '/^(\(?)\s*([-+]?[0-9]*\.?[0-9]+)\s*,' .
                    '\s*([-+]?[0-9]*\.?[0-9]+)\s*(\)?)$/',
                $value,
                $m
            );

            return (
                $valid
                &&
                (strlen($m[1]) === strlen($m[4]))
                &&
                is_numeric($m[2])
                &&
                is_numeric($m[3])
            );
        }

        return false;
    }

    /**
     * Decode a string to an array
     *
     * @param string $string
     *
     * @return void
     */

    private function _decode($value)
    {
        if (is_array($value))
        {
            $this->_x = $value[0];
            $this->_y = $value[1];
        }
        elseif (is_string($value))
        {
            preg_match(
                '/^\(?\s*([-+]?[0-9]*\.?[0-9]+)\s*,' .
                    '\s*([-+]?[0-9]*\.?[0-9]+)\s*\)?$/',
                $value,
                $m
            );

            $this->_x = $m[1];
            $this->_y = $m[2];
        }
    }

    /**
     * __set
     *
     * The magic setter (including constraint on attributes)
     *
     * @param string $name
     * @param mixed  $value
     *
     * @return void
     */

    public function __set($name, $value)
    {
        if (is_string($name) && in_array($name, array('x', 'y')))
        {
            $this->{"_$name"} = $value;
        }
    }

    /**
     * __get
     *
     * The magic getter
     *
     * @param string $name
     *
     * @return mixed
     */

    public function __get($name)
    {
        if (is_string($name) && in_array($name, array('x', 'y')))
        {
            return $this->{"_$name"};
        }

        return null;
    }

    /**
     * __toString
     *
     * Returns the string version of the hstore
     *
     * @return string
     */

    public function __toString()
    {
        return '( ' . $this->_x . ' , ' . $this->_y . ' )';
    }

    /**
     * Compare 2 point on a given precision
     * @param Point $point
     * @param int $precision
     * @return boolean
     */
    public function isEqualTo(Point $point, $precision = 5)
    {
        $xDec = strlen($this->getDecimalValue($this->_x));
        $yDec = strlen($this->getDecimalValue($this->_y));
        $xDecPoint = strlen($this->getDecimalValue($point->_x));
        $yDecPoint = strlen($this->getDecimalValue($point->_y));

        // Check if precision is enough to continue
        if(
            $xDec < $precision || $yDec < $precision ||
            $xDecPoint < $precision || $yDecPoint < $precision
        )
        {
            return false;
        }

        $length = min($xDec, $yDec, $xDecPoint, $yDecPoint);

        return
            $this->getRoundValue($this->_x, $length) ==
                $this->getRoundValue($point->_x, $length) &&
            $this->getRoundValue($this->_y, $length) ==
                $this->getRoundValue($point->_y, $length);
    }

    /**
     * Return decimal value of a float number
     * @param type $number
     * @return type
     */
    protected function getDecimalValue($number)
    {
        $dec = floatval($number) - floor($number);

        return $dec * pow(10, strlen($dec) - 2);
    }

    /**
     * Return round value
     * @param float $number
     * @param int $length
     * @return float
     */
    protected function getRoundValue($number, $length)
    {
        return (int) ($number * pow(10, $length)) / pow(10, $length);
    }
}