<?php

namespace Orms\Statement;

/**
 * Classe de base pour les Statements ORMs
 *
 * @author Xavier HAUSHERR <xavier.hausherr@ebuzzing.com>
 */
abstract class Statement implements \Iterator, \Countable
{
    /**
     * Pour garder la compatibilité avec PDO
     */
    const FETCH_ASSOC = 2;
    const FETCH_OBJ = 5;
    const FETCH_NUM = 3;

    /**
     * Query resource
     * @var ressource
     */
    protected $resource;

    /**
     * Query ID
     * @var string
     */
    protected $id;

    /**
     * Connection Ressource
     * @var ressource
     */
    protected $connectionRessource;

    /**
     * Iterator key
     * @var int
     */
    protected $key = 0;

    /**
     * Init Statement
     * @param ressource $resource
     * @param string $id
     * @param ressource $connectionRessource
     */
    public function __construct($resource, $id = null, $connectionRessource = null)
    {
        $this->resource = $resource;

        if(!is_null($id))
        {
            $this->id = $id;
        }

        if(!is_null($connectionRessource))
        {
            $this->connectionRessource = $connectionRessource;
        }
    }

    /**
     * Compte le nombre d'éléments
     * @return int
     */
    public function count()
    {
        return $this->rowCount();
    }

    /**
     * Retourne l'élément courant
     *
     * @return mixed
     */
    public function current()
    {
        return $this->fetch(self::FETCH_ASSOC, $this->key());
    }

    /**
     * Retourne la clé de l'élément courant
     * @return int
     */
    public function key()
    {
        return $this->key;
    }

    /**
     * Se déplace sur l'élément suivant
     */
    public function next()
    {
        $this->key++;
    }

    /**
     * Replace l'itérateur sur le premier élément
     */
    public function rewind()
    {
        $this->key = 0;
    }

    /**
     * Vérifie si la position courante est valide
     *
     * @return boolean
     */
    public function valid()
    {
        return $this->key() < $this->count();
    }

    /**
     * Retourne le nombre de lignes PostgreSQL
     *
     * @return int
     */
    abstract public function rowCount();

    /**
     * Récupère la ligne suivante d'un jeu de résultats
     *
     * @param int $style
     * @param int $index
     * @return mixed
     */
    abstract public function fetch($style = self::FETCH_ASSOC, $index = null);

    /**
     * Exécute une requête préparée
     *
     * @param array $params
     * @return boolean
     */
    abstract public function execute($params);

    /**
     * Libère la mémoire
     *
     * @return boolean
     */
    abstract public function __destruct();
}
