<?php

namespace Orms\DataProvider;

use Orms\Exception\Exception;
use Orms\Exception\QueryException;
use Orms\DataProvider\GenericDataProvider;
use Orms\Statement\Statement;

abstract class PostgresqlDataProvider extends GenericDataProvider
{
    /**
     * shema & table name
     */

    protected $_schema;
    protected $_table;

    /**
     * @param mixed $options
     */

    public function setup($options)
    {
        if (!array_key_exists('schema', $options)) {
            throw new Exception(
                "PostgresqlDataProvider : \$options['schema'] is missing."
            );
        }

        if (!array_key_exists('table', $options)) {
            throw new Exception(
                "PostgresqlDataProvider : \$options['table'] is missing."
            );
        }

        $this->_schema = $options['schema'];
        $this->_table = $options['table'];
    }

    /**
     * Inserts some data.
     *
     * @param array $definition
     * @param array $values
     *
     * @return mixed - returns array of $values after the insertion || throw an
     * exception
     *
     * @author Yannick Le Guédart
     */

    public function insert(
        array $definition,
        array $values
    ) {
        // génération des noms de colonnes

        $columns = [];
        $_values = [];
        $i = 1;

        foreach ($definition['columns'] as $name => $def) {
            // Si c'est une clef primaire et qu'elle est à null, on l'ignore,
            // ainsi que si elle est à null et qu'elle est indiqué comme ayant
            // une valeur par défaut

            if (
                is_null($values[$name])
                &&
                (
                    in_array($name, $definition['pkFields'])
                    ||
                    (
                            isset($def['has_default'])
                            &&
                            ($def['has_default'] === true)
                    )
                )
            ) {
                continue;
            }

            // gestion des booléens

            if (
                array_key_exists('type', $def)
                &&
                ($def['type'] === 'boolean')
                &&
                is_bool($values[$name])
            ) {
                $values[$name] = ($values[$name] === true) ? 't' : 'f';
            }

            // préparation des données

            $columns[] = '"' . $name . '"';
            $preparing[] = '$' . $i++;
            $_values[] = $values[$name];
        }

        // Génération de la requète d'insertion

        $sql = trim('INSERT INTO ' . $this->_getTableName() .
                    '(' . implode(',', $columns) . ') ' .
                    'VALUES ' .
                    '(' . implode(',', $preparing) . ') ' .
                    'RETURNING *;');

        // Execution

        $logger      = self::getLogger();
        self::$_meta[get_called_class()] = new \StdClass();

        if ($logger) {
            $logger->startQuery($sql, $_values);
        }

        $sth = $this->getConnection()->prepare($sql);

        if ($sth->execute($_values)) {
            self::$_meta[get_called_class()]->rowCount = $sth->rowCount();

            if ($logger) {
                $logger->stopQuery(
                    false,
                    self::$_meta[get_called_class()]->rowCount
                );
            }

            return $sth->fetch(Statement::FETCH_ASSOC);
        } else {
            $e = new QueryException($sth);

            if ($logger) {
                $logger->logException($e->getErrorInfo());
            }

            self::$_meta[get_called_class()]->errorCode = $sth->errorCode();
            self::$_meta[get_called_class()]->errorInfo = $sth->errorInfo();

            throw $e;
        }
    }

    /**
     * Update a row.
     *
     * @param array $definition
     * @param array $values
     *
     * @return mixed - returns array of $values after the insertion || throw an
     * exception
     *
     * @author Yannick Le Guédart
     */

    public function update(
        array $definition,
        array $values
    ) {
        // verifying that primary keys are set

        $pks = [];

        foreach ($definition['pkFields'] as $pk) {
            if (is_null($values[$pk])) {
                throw new Exception(
                    "Primary key $pk not set on update"
                );
            }

            $pks[] = '"' . $pk . '" = ' . $this->getConnection()->quote($values[$pk]);
        }

        // génération des noms de colonnes

        $columns = [];
        $_values = [];
        $i = 1;

        foreach ($definition['columns'] as $name => $def) {
            // gestion des booléens

            if (
                array_key_exists('type', $def)
                &&
                ($def['type'] === 'boolean')
                &&
                is_bool($values[$name])
            ) {
                $values[$name] = ($values[$name] === true) ? 't' : 'f';
            }

            // préparation des données

            $preparingColumns[] = '"' . $name . '" = $' . $i++;
            $_values[] = $values[$name];
        }

        // Génération de la requète d'insertion

        $sql = trim('UPDATE ' . $this->_getTableName() . ' SET ' .
                    implode(',', $preparingColumns) .
                    ' WHERE ' . implode(' AND ', $pks) .
                    ' RETURNING *;');

        // Execution

        $logger      = self::getLogger();
        self::$_meta[get_called_class()] = new \StdClass();

        if ($logger) {
            $logger->startQuery($sql, $_values);
        }

        $sth = $this->getConnection()->prepare($sql);

        if ($sth->execute($_values)) {

            self::$_meta[get_called_class()]->rowCount = $sth->rowCount();

            if ($logger) {
                $logger->stopQuery(
                    false,
                    self::$_meta[get_called_class()]->rowCount
                );
            }


            return $sth->fetch(Statement::FETCH_ASSOC);
        } else {
            $e = new QueryException($sth);

            if ($logger) {
                $logger->logException($e->getErrorInfo());
            }

            self::$_meta[get_called_class()]->errorCode = $sth->errorCode();
            self::$_meta[get_called_class()]->errorInfo = $sth->errorInfo();

            throw $e;
        }
    }

    /**
     * Delete a row.
     *
     * @param array $definition
     * @param array $values
     *
     * @return boolean
     *
     * @author Yannick Le Guédart
     */

    public function delete(
        array $definition,
        array $values
    ) {
        // verifying that primary keys are set

        $pks = [];

        foreach ($definition['pkFields'] as $pk) {
            if (is_null($values[$pk])) {
                throw new Exception(
                    "Primary key $pk not set on update"
                );
            }

            $pks[] = '"' . $pk . '" = ' . $this->getConnection()->quote($values[$pk]);
        }

        // Génération de la requète d'insertion

        $sql = trim('DELETE FROM ' . $this->_getTableName() . ' ' .
                    'WHERE ' . implode(' AND ', $pks));

        // Execution

        $logger      = self::getLogger();
        self::$_meta[get_called_class()] = new \StdClass();

        if ($logger) {
            $logger->startQuery($sql, $values);
        }

        $result = $this->getConnection()->exec($sql);

        self::$_meta[get_called_class()]->rowCount = $result;

        if ($logger) {
            $logger->stopQuery(
                false,
                self::$_meta[get_called_class()]->rowCount
            );
        }

        return (bool) $result;
    }


    public function get(array $params)
    {
        $params['_tableName'] = $this->_getTableName();
        $params['_connection'] = $this->getConnection();

        return parent::get($params);
    }

    /**
     * Returns the full tablename in the postgresqldatabase
     *
     * @return string
     */

    protected function _getTableName()
    {
        return
            (
                (
                    is_null($this->_schema)
                        ?
                        ''
                        :
                        ('"' . $this->_schema . '".')
                )
                .
                '"' . $this->_table . '"'
            );
    }

    /**
     * get composer
     */

    public static $_getBaseParams =

        [
            '_tableName' => null,
            '_connection' => null
        ];

    public static $_getFilterParams =

        [
            'id'       => null,
            'id_array' => null
        ];

    protected static function getComposer($params)
    {
        if (!is_array($params)) {
            throw new Exception(
                '$params is not an array'
            );
        }

        // Tableaux de paramètres par défaut.

        $baseDefault   = [];
        $filterDefault = [];

        // Tableaux de paramètres réels.

        $filterParams = [];

        $classLoop = get_called_class();

        while ($classLoop) {
            foreach ($classLoop::computeBaseParams() as $k => $v) {
                if (!isset($baseDefault[$k])) {
                    $baseDefault[$k] = $v;
                }
            }

            foreach ($classLoop::computeFilterParams() as $k => $v) {
                if (!isset($filterDefault[$k])) {
                    $filterDefault[$k] = $classLoop;

                    if (!is_null($v)) {
                        $filterParams[$k] = $v;
                    }
                }
            }

            if ($classLoop === 'GenericDataProvider') {
                break;
            }

            $classLoop = get_parent_class($classLoop);
        }

        /* ---------------------------------------------------------------------
           * Génération des tableaux de paramètres réels
           * ---------------------------------------------------------------------
           */

        $baseParams = $baseDefault;

        foreach ($params as $k => $v) {
            if (array_key_exists($k, $baseParams)) {
                $baseParams[$k] = $v;
            } elseif (array_key_exists($k, $filterDefault)) {
                $filterParams[$k] = $v;
            } else {
                throw new \Orms\Exception\Exception(
                    'Invalid query parameter [' . $k . ']'
                );
            }
        }

        /* ---------------------------------------------------------------------
           * Génération du tableau de bouts de requetes SQL utilisées poru les
           * filtres.
           * ---------------------------------------------------------------------
           */

        $t = $baseParams['_tableName'];
        $db = $baseParams['_connection'];

        $filterSQLArray = [];

        foreach ($filterParams as $k => $v) {
            if (!is_null($v)) {
                $oneFilter =
                    call_user_func_array(

                        [
                            $filterDefault[$k],
                            'getFilterSQL'
                        ],
                        [$k, $v, $t, $db]
                    );

                if (!is_null($oneFilter) && ($oneFilter !== '')) {
                    $filterSQLArray[] = $oneFilter;
                }
            }
        }

        if (count($filterSQLArray) > 0) {
            $filterSQL =
                "WHERE " . implode(' AND ', array_unique($filterSQLArray));
        } else {
            $filterSQL = '';
        }

        /* ---------------------------------------------------------------------
         * Gestion des paramètres par défaut.
         *
         * C'est ici qu'on va gérer les options par défaut.
         *
         * Tout d'abord, $baseParams['countOnly'] détermine si l'ont fait un
         * décompte ou si l'on veut des éléments. Dans le cas d'un décompte, on
         * peut ignorer tous les paramètres du tableau de base, mais aussi tous
         * paramètres de champs supplémentaires.
         *
         * On commence à construire la chaine des champs à recupérer dans la
         * requète, la chaine liée à l'order by, ainsi que les offset et limit.
         * ---------------------------------------------------------------------
         */

        $orderSQL  = null;
        $offsetSQL = '';
        $groupSQL  = '';
        $limitSQL  = '';
        $_savable  = true;

        if ($baseParams['countOnly'] === true) {
            if (!is_null($baseParams['_countBy'])) {
                $fieldSQL = [sprintf("COUNT (DISTINCT %s.%s) AS c", $t, $baseParams['_countBy'])];
            } else {
                $fieldSQL = ["COUNT ($t.*) AS c"];
            }
        } else {
            /* -----------------------------------------------------------------
             * Offset / Limit
             *
             * Un offset de 0 ou un offset non numérique est ignoré.
             * -----------------------------------------------------------------
             */

            if (is_numeric($baseParams['offset'])
                && ($baseParams['offset'] > 0)
            ) {
                $offsetSQL = 'OFFSET ' . $baseParams['offset'];
            }

            if (!is_null($baseParams['limit'])) {
                $limitSQL = 'LIMIT ' . $baseParams['limit'];
            }

            /** Handling order */

            if (! is_null($baseParams['order'])) {
                if (! is_array($baseParams['order'])) {
                    throw new \Orms\Exception\Exception(
                        'Invalid order parameter.'
                    );
                } else {
                    $orderSQLArray = [];

                    foreach ($baseParams['order'] as $field => $asc) {
                        $orderSQLArray[] =
                            "$field " . ((true === $asc) ? 'ASC' : 'DESC');
                    }

                    $orderSQL = 'ORDER BY ' . implode(', ', $orderSQLArray);
                }
            } elseif (!is_null($baseParams['orderType'])) {
                $orderSQL =
                    'ORDER BY ' . $baseParams['orderType'] . ' ' .
                        (($baseParams['orderAsc'] === true) ? 'ASC' : 'DESC');
            }

            if (
                is_array($baseParams['_onlyFields'])
                &&
                (count($baseParams['_onlyFields']) > 0)
            ) {
                $fieldSQL = [];

                foreach ($baseParams['_onlyFields'] as $field) {
                    $fieldSQL[] = "$t.$field";
                }

                $_savable = false;
            } else {
                $fieldSQL = ["$t.*"];
            }
        }

        if (is_array($baseParams['_groupBy'])) {
            $groupSQL = 'GROUP BY ' . implode(',', $baseParams['_groupBy']);
        }

        /**
         * On injecte la variable _savable dans la requète
         * pour l'avoir directement dans l'objet
         */
        $fieldSQL[] = sprintf('%s::BOOL AS _savable', (int)$_savable);

        /* ---------------------------------------------------------------------
         * Création de la requête SQL.
         * ---------------------------------------------------------------------
         */

        $sql =
            trim("SELECT " . join(',', $fieldSQL) .
            " FROM $t " .
            "$filterSQL " .
            "$groupSQL " .
            "$orderSQL " .
            "$offsetSQL " .
            "$limitSQL");

        /* ---------------------------------------------------------------------
         * Exécution de la requête SQL.
         *
         * En fonction du type de requète, $baseParams['countOnly'] ou pas, on
         * revoit un entier ou un ArrayObject.
         * ---------------------------------------------------------------------
         */

        $logger = self::getLogger();

        if ($logger) {
            $logger->startQuery($sql, $params);
        }

        $rs = $db->query($sql);

        self::$_meta[get_called_class()] = new \StdClass();
        // If PDO object is empty, then query id corrupt

        if (false === $rs) {
            $e = new QueryException($db);

            if ($logger) {
                $logger->logException($e->getErrorInfo());
            }

            self::$_meta[get_called_class()]->errorCode = $db->errorCode();
            self::$_meta[get_called_class()]->errorInfo = $db->errorInfo();

            throw $e;
        }

        if ($baseParams['countOnly']) {
            $record = $rs->fetch(Statement::FETCH_ASSOC);

            self::$_meta[get_called_class()]->rowCount = 1;

            if ($logger) {
                $logger->stopQuery(false, 1);
            }

            return intval($record['c']);
        } else {
            // Save result count in meta
            self::$_meta[get_called_class()]->rowCount = $rs->rowCount();

            if ($logger) {
                $logger->stopQuery(
                    false,
                    self::$_meta[get_called_class()]->rowCount
                );
            }
            return $rs;
        }
    }

    public static function getFilterSQL($k, $v, $t, $db)
    {
        return static::_getFilterSQL($k, $v, $t, $db);
    }

    /**
     * @deprecated 2.2.0 use getFilterSQL instead
     */
    public static function _getFilterSQL($k, $v, $t, $db)
    {
        $filterSQL = null;

        switch ($k) {
            case 'id':
                $filterSQL = "$t.id = " . $db->quote($v);
                break;
            case 'id_array':
                if (! is_array($v) || empty($v)) {
                    $filterSQL = 'FALSE';
                } else {
                    foreach ($v as $k => $value) {
                        $v[$k] = $db->quote($value);
                    }
                    $filterSQL = "$t.id IN (" . implode(',', $v) . ')';
                }
                break;
        }

        return $filterSQL;
    }
}
