<?php

namespace Orms\Connection;

use Orms\Connection\SqlInterface;
use Orms\Exception\DatabaseException;
use Orms\Statement\MysqlStatement;

class Mysql implements SqlInterface
{
    /**
     * MySQL Connection handler
     * @var ressource
     */
    protected $connection;

    /**
     * Ouvre une nouvelle connection MySQL
     *
     * @param array $options
     * @throws \Orms\Exception\Exception
     */
    public function __construct($options = [])
    {
        $this->connection = @mysqli_connect(
                $options['host'],
                $options['username'],
                $options['password'],
                $options['dbname'],
                (isset($options['port'])) ? $options['port'] : 3306
            );

        if (mysqli_connect_errno()) {
            throw new DatabaseException(
                sprintf(
                    'Unable to connect to MySQL server: %s',
                    mysqli_connect_error()
                )
            );
        }

        if (array_key_exists('charset', $options)) {
            mysqli_set_charset($this->connection, $options['charset']);
        }
    }

    /**
     * {@inheritDoc}
     */
    public function __destruct()
    {
        return @mysqli_close($this->connection);
    }

    /**
     * {@inheritDoc}
     */
    public function errorCode()
    {
        return mysqli_errno($this->connection);
    }

    /**
     * {@inheritDoc}
     */
    public function errorInfo()
    {
        return mysqli_error($this->connection);
    }

    /**
     * {@inheritDoc}
     */
    public function exec($sql)
    {
        $result = @mysqli_query($this->connection, $sql);

        if (false !== $result) {
            return mysqli_affected_rows($this->connection);
        } else {
            return false;
        }
    }

    /**
     * {@inheritDoc}
     */
    public function ping()
    {
        return mysqli_ping($this->connection);
    }

    /**
     * {@inheritDoc}
     */
    public function prepare($sql)
    {
        $result = mysqli_prepare($this->connection, $sql);

        if (false !== $result) {
            return new MysqlStatement($result, null, $this->connection);
        } else {
            return false;
        }
    }

    /**
     * {@inheritDoc}
     */
    public function query($sql)
    {
        $result = @mysqli_query($this->connection, $sql);

        if (false !== $result) {
            return new MysqlStatement($result);
        } else {
            return false;
        }
    }

    /**
     * {@inheritDoc}
     */
    public function quote($data)
    {
        return "'" . mysqli_real_escape_string($this->connection, $data) . "'";
    }

    /**
     * {@inheritDoc}
     */
    public function beginTransaction()
    {
        return $this->connection->begin_transaction();
    }

    /**
     * {@inheritDoc}
     */
    public function commit()
    {
        return $this->connection->commit();
    }

    /**
     * {@inheritDoc}
     */
    public function rollback()
    {
        return $this->connection->rollback();
    }
}
