<?php

namespace Orms\Connection;

use Orms\Connection\SqlInterface;
use Orms\Exception\DatabaseException;
use Orms\Statement\MssqlStatement;

class Mssql implements SqlInterface
{
    /**
     * @var \PDO
     */
    protected $connection;

    /**
     * Ouvre une nouvelle connection Mssql
     *
     * @param  array $options
     *
     * @throws DatabaseException
     */
    public function __construct($options = array())
    {
        $dsn = sprintf(
            "dblib:host=%s;dbname=%s",
            $options['host'] . (isset($options['port']) ? (' : ' . $options['port']) : ''),
            $options['dbname']
        );

        $this->connection = new \PDO($dsn, $options['username'], $options['password']);

        if (!$this->connection) {
            throw new DatabaseException(
                'Unable to connect to MsSQL server. Reason : ' .
                $this->connection->errorInfo()
            );
        }
    }

    /**
     * {@inheritDoc}
     */

    public function ping()
    {
        return (false !== $this->query('SELECT 1;'));
    }

    /**
     * {@inheritDoc}
     */

    public function quote($data)
    {
        if (is_null($data)) {
            return 'NULL';
        }

        /** There we should have a string */

        return
            "'" .
            preg_replace(
                [
                    "/'/",
                    '/"/',
                    '/%0[0-8bcef]/', // url encoded 00-08, 11, 12, 14, 15
                    '/%1[0-9a-f]/',  // url encoded 16-31
                    '/[\x00-\x08]/', // 00-08
                    '/\x0b/',        // 11
                    '/\x0c/',        // 12
                    '/[\x0e-\x1f]/'  // 14-31
                ],
                [
                    "''",
                    '""',
                    '',
                    '',
                    '',
                    '',
                    '',
                    ''
                ],
                $data
            ) .
            "'";
    }

    /**
     * {@inheritDoc}
     */
    public function query($sql)
    {
        $result = $this->connection->query($sql);

        return $result !== false ? new MssqlStatement($this->connection, $result) : false;
    }

    /**
     * {@inheritDoc}
     */
    public function prepare($sql)
    {
        $result = $this->connection->prepare($sql);

        return $result !== false ? new MssqlStatement($this->connection, $result) : false;
    }

    /**
     * {@inheritDoc}
     */
    public function exec($sql)
    {
        return $this->connection->exec($sql);
    }

    /**
     * {@inheritDoc}
     */
    public function errorCode()
    {
        return $this->connection->errorInfo()[1];
    }

    /**
     * {@inheritDoc}
     */
    public function errorInfo()
    {
        $info = $this->connection->errorInfo();

        return sprintf('%s (%s)', $info[2], $info[0]);
    }

    /**
     * {@inheritDoc}
     */
    public function __destruct()
    {
        $this->connection = null;
    }
}
