<?php

use Orms\Exception\Exception;
use Orms\Test\Mock\ConcreteConnection;
use Orms\Test\Mock\ConcreteDataObject;
use Orms\Test\Mock\ConcreteDataObjectNotSavableClass;
use Orms\Test\Mock\ConcreteDataProvider;
use Orms\Type\ArrayObject;

class DataObjectTest extends \PHPUnit_Framework_TestCase
{
    public function testConstruct2()
    {
        $o =
			new ConcreteDataObject(
				new ConcreteDataProvider(new ConcreteConnection())
			);

		$values = $o->getValues();

		$this->assertArrayHasKey('col1', $values);
		$this->assertArrayHasKey('col2', $values);
		$this->assertArrayHasKey('col3', $values);
		$this->assertArrayHasKey('col4', $values);
		$this->assertArrayHasKey('col5', $values);
		$this->assertArrayHasKey('col6', $values);

		// Testing defaults

		$this->assertNull($values['col1']);
		$this->assertNull($values['col2']);
		$this->assertEquals('', $values['col3']);
		$this->assertEquals(
			"<?xml version=\"1.0\"?>\n<data/>\n",
			$values['col4']
		);
		$this->assertEquals(date('Y-m-d'), $values['col5']);
		$this->assertEquals('{}', $values['col6']);

    }

    public function testConstruct3()
    {
        $o =
			new ConcreteDataObject(
				new ConcreteDataProvider(new ConcreteConnection()),
				42
			);

		$values = $o->getValues();

		$this->assertEquals(42, $values['col1']);
		$this->assertEquals('col2 42', $values['col2']);
		$this->assertEquals('"col3"=>"42"', $values['col3']);
		$this->assertEquals('<data><col4>42</col4></data>', $values['col4']);
		$this->assertEquals('2007-01-01 02:02:02', $values['col5']);
		$this->assertEquals('{col6,42}', $values['col6']);
	}

    public function testConstruct4()
    {
        $this->setExpectedException(
			'Orms\Exception\NoRowsFoundException',
			"Can't find Orms\Test\Mock\ConcreteDataObject object in data provider with " .
				"the primary key [43]"
		);

		$o = new ConcreteDataObject(new ConcreteDataProvider(new ConcreteConnection()), 43);
	}

    public function testConstruct5()
    {
        $o = new ConcreteDataObject(new ConcreteDataProvider(new ConcreteConnection()), array('col1' => 42));

		$values = $o->getValues();

		$this->assertEquals(42, $values['col1']);
		$this->assertEquals('col2 42', $values['col2']);
		$this->assertEquals('"col3"=>"42"', $values['col3']);
		$this->assertEquals('<data><col4>42</col4></data>', $values['col4']);
		$this->assertEquals('2007-01-01 02:02:02', $values['col5']);
		$this->assertEquals('{col6,42}', $values['col6']);
	}

    public function testConstruct6()
    {
        $this->setExpectedException(
			'Orms\Exception\NoRowsFoundException',
			"Can't find Orms\Test\Mock\ConcreteDataObject object in data provider with " .
				"the params [Array
(
    [col1] => 43
    [limit] => 2
)
]"
		);

        $o = new ConcreteDataObject(new ConcreteDataProvider(new ConcreteConnection()), array('col1' => 43));
	}

    public function testConstruct7()
    {
		$object =
			(object)
			array(
				'col1'		=> 42,
				'col2'		=> 'col2 42',
				'col3'		=> '"col3"=>"42"',
				'col4'		=> '<data><col4>42</col4></data>',
				'col5'		=> '2007-01-01 02:02:02',
				'col6'		=> '{col6,42}',
			);

        $o = new ConcreteDataObject(new ConcreteDataProvider(new ConcreteConnection()), $object);

		$values = $o->getValues();

		$this->assertEquals(42, $values['col1']);
		$this->assertEquals('col2 42', $values['col2']);
		$this->assertEquals('"col3"=>"42"', $values['col3']);
		$this->assertEquals('<data><col4>42</col4></data>', $values['col4']);
		$this->assertEquals('2007-01-01 02:02:02', $values['col5']);
		$this->assertEquals('{col6,42}', $values['col6']);
	}

    public function testConstruct8()
    {
        $this->setExpectedException(
			'Orms\Exception\Exception',
			"Invalid data type [boolean]"
		);

        $o = new ConcreteDataObject(new ConcreteDataProvider(new ConcreteConnection()), true);
	}

    public function testSerializable()
    {
		$array =
			array(
				'col1'		=> 42,
				'col2'		=> 'col2 42',
				'col3'		=> '"col3"=>"42"',
				'col4'		=> '<data><col4>42</col4></data>',
				'col5'		=> '2007-01-01 02:02:02',
				'col5_1'	=> '2008-01-01 02:02:02',
				'col5_2'	=> '2009-01-01 02:02:02',
				'col6'		=> '{col6,42}',
			);

        $o =
			new ConcreteDataObject(
				new ConcreteDataProvider(new ConcreteConnection()),
				(object) $array
			);

		$values = $o->getValues();

		$this->assertEquals(serialize($array), $o->serialize());

		$o2 =
			new ConcreteDataObject(
				new ConcreteDataProvider(new ConcreteConnection())
			);

		$this->assertEquals(
			$array,
			$o2->unserialize($o->serialize())->getValues()
		);

	}


    public function testIsset1()
    {
        $o = new ConcreteDataObject(new ConcreteDataProvider(new ConcreteConnection()));

		$this->assertTrue(isset($o->col1));
		$this->assertTrue(isset($o->col2));
		$this->assertTrue(isset($o->col3));
		$this->assertTrue(isset($o->col4));
		$this->assertTrue(isset($o->col5));
		$this->assertTrue(isset($o->col6));

		$this->assertFalse(isset($o->col9));
		$this->assertFalse(isset($o->col51));
	}

    public function testGetSet1()
    {
        $o = new ConcreteDataObject(new ConcreteDataProvider(new ConcreteConnection()));

        $this->setExpectedException(
			'Orms\Exception\Exception',
			'Property [col51] does not exist.'
		);

		$o->col51;
	}

    public function testGetSet2()
    {
        $o = new ConcreteDataObject(new ConcreteDataProvider(new ConcreteConnection()));

		$this->assertNull($o->col1);
		$this->assertNull($o->col2);

		$this->assertEquals(new \Orms\Type\Hstore(''), $o->col3);
		$this->assertEquals(
			new \Orms\Tools\SimpleXMLElementExtended('<data />'),
			$o->col4
		);
		$this->assertEquals(new \DateTime('now'), $o->col5);
		$this->assertEquals(new ArrayObject(), $o->col6);

		$o->col1 = 12;
		$o->col2 = 'Bidule';

		$this->assertEquals(12, $o->col1);
		$this->assertEquals('Bidule', $o->col2);

		$o->col3->youpi = 'TRUC';

		$this->assertEquals(
			new \Orms\Type\Hstore('"youpi"=>"TRUC"'),
			$o->col3
		);

		$o->col4->youpi = 'TRUC';

		$this->assertEquals(
			new \Orms\Tools\SimpleXMLElementExtended(
				'<data><youpi>TRUC</youpi></data>'
			),
			$o->col4
		);

        $this->setExpectedException(
			'Orms\Exception\Exception',
			'Property [col51] does not exist.'
		);

		$o->col51 = 'bidule';
	}

    public function testGetSet3()
    {
        $o = new ConcreteDataObject(new ConcreteDataProvider(new ConcreteConnection()));

        $this->setExpectedException(
			'Orms\Type\Exception',
			"Can't set a Orms\Type\Hstore object with the value [bidule]"
		);

		$o->col3 = 'bidule';
	}

    public function testSetFrom1()
    {
        $o = new ConcreteDataObject(new ConcreteDataProvider(new ConcreteConnection()));

        $this->setExpectedException(
			'Orms\Exception\Exception',
			"setFrom method first parameter must be array || object, string given."
		);

		$o->setFrom('bidule');
	}

    public function testSetFrom2()
    {
        $o = new ConcreteDataObject(new ConcreteDataProvider(new ConcreteConnection()));

        $this->setExpectedException(
			'Orms\Exception\Exception',
			"setFrom method second parameter must be array || object, string given."
		);

		$o->setFrom(array(), 'bidule');
	}

    public function testSetFrom3()
    {
		$object =
			(object)
			array(
				'col1'		=> 42,
				'col2'		=> 'col2 42',
				'col3'		=> '"col3"=>"42"',
				'col4'		=> '<data><col4>42</col4></data>',
				'col5'		=> '2007-01-01 02:02:02',
				'col6'		=> '{col6,42}',
				'col7'		=> '1111000011110000',
				'col8'		=> 42,
				'col9'		=> 42,
				'col51'		=> 43
			);

        $o = new ConcreteDataObject(new ConcreteDataProvider(new ConcreteConnection()));

		$o->setFrom($object, array('col1', 'col6'));

		$values = $o->getValues();

		// Except

		$this->assertEquals(null, $values['col1']);
		$this->assertEquals('{}', $values['col6']);

		// Normal

		$this->assertEquals('col2 42', $values['col2']);
		$this->assertEquals('"col3"=>"42"', $values['col3']);
		$this->assertEquals('<data><col4>42</col4></data>', $values['col4']);
		$this->assertEquals('2007-01-01 02:02:02', $values['col5']);

		// External

		$this->assertEquals(42, $o->external->col9);
		$this->assertEquals(43, $o->external->col51);
	}

	/**
	 * @group GLOU
	 */

    public function testFind1()
    {
        $os =
			ConcreteDataObject::find(
				new ConcreteDataProvider(new ConcreteConnection()),
				array('col1' => 42)
			);

		$this->assertEquals(1, $os->count());

	    $o = $os->offsetGet(0);

		$values = $o->getValues();

		$this->assertEquals(42, $values['col1']);
		$this->assertEquals('col2 42', $values['col2']);
		$this->assertEquals('"col3"=>"42"', $values['col3']);
		$this->assertEquals('<data><col4>42</col4></data>', $values['col4']);
		$this->assertEquals('2007-01-01 02:02:02', $values['col5']);
		$this->assertEquals('{col6,42}', $values['col6']);
	}

    public function testSave1() // insert
    {
        $o = new ConcreteDataObject(new ConcreteDataProvider(new ConcreteConnection()));

		$o->save();

		$values = $o->getValues();

		$this->assertEquals(43, $values['col1']);
		$this->assertEquals('col2 42', $values['col2']);
		$this->assertEquals('"col3"=>"42"', $values['col3']);
		$this->assertEquals('<data><col4>42</col4></data>', $values['col4']);
		$this->assertEquals('2007-01-01 02:02:02', $values['col5']);
		$this->assertEquals('{col6,42}', $values['col6']);
	}

    public function testSave2() // update
    {
        $o = new ConcreteDataObject(new ConcreteDataProvider(new ConcreteConnection()), 42);

		$this->assertInstanceOf('Orms\Type\Hstore', $o->col3);
		$this->assertInstanceOf('Orms\Tools\SimpleXMLElementExtended', $o->col4);
		$this->assertInstanceOf('\DateTime', $o->col5);
		$this->assertInstanceOf('\DateTime', $o->col5_1);
		$this->assertInstanceOf('\DateTime', $o->col5_2);
		$this->assertInstanceOf('\ArrayObject', $o->col6);

		$this->assertTrue($o->save());

		$values = $o->getValues();

		$this->assertEquals(44, $values['col1']);
		$this->assertEquals('col2 42', $values['col2']);
		$this->assertEquals('"col3"=>"42"', $values['col3']);
		$this->assertEquals('<data><col4>42</col4></data>', $values['col4']);
		$this->assertEquals('2007-01-01 02:02:02', $values['col5']);
		$this->assertEquals('{col6,42}', $values['col6']);
	}

    public function testSave3() // update
    {
        $o =
			new ConcreteDataObjectNotSavableClass(
				new ConcreteDataProvider(new ConcreteConnection())
			);

        $this->setExpectedException(
			'Orms\Exception\Exception',
			"Object cannot be saved"
		);

		$o->save();
	}

	public function testDelete1()
	{
		$o = new ConcreteDataObject(new ConcreteDataProvider(new ConcreteConnection()), 42);

		$this->assertTrue($o->delete());
	}

    public function testHasBeenModified()
    {
        $o = new ConcreteDataObject(new ConcreteDataProvider(new ConcreteConnection()), 42);

        $this->assertFalse($o->hasBeenModified());

        $o->col2 = "TEST";

        $this->assertTrue($o->hasBeenModified());

        $o->save();

        $this->assertFalse($o->hasBeenModified());
    }

    public function testHasBeenModifiedWithSerialize()
    {
        $o = new ConcreteDataObject(new ConcreteDataProvider(new ConcreteConnection()), 42);

        $this->assertFalse($o->hasBeenModified());

        $oSerialize = serialize($o);

        $o2 = unserialize($oSerialize);

        $this->assertFalse($o2->hasBeenModified());
    }
}