<?php

namespace Orms\Proxy;

use Orms\Proxy\Dumper\MethodDumperInterface;
use Orms\Proxy\Dumper\OrmsConstructorDumper;
use Orms\Proxy\Dumper\ProxyDumper;
use Orms\Proxy\Registry;

class Manager
{
    private $strategiesMap = [
        'direct'   => 'Orms\Proxy\Dumper\DirectMethodDumper',
        'uncached' => 'Orms\Proxy\Dumper\UncachedDirectMethodDumper',
        'fetcher'  => 'Orms\Proxy\Dumper\FetcherMethodDumper'
    ];

    /**
     * @var string
     */
    private $path;

    /**
     * @var boolean
     */
    private $debug;

    /**
     * @var ProxyDumper
     */
    private $dumper;

    /**
     * @var MethodDumperInterface[]
     */
    private $dumpers;

    /**
     * @var string[]
     */
    private $classes;

    /**
     * @param ProxyDumper $dumper
     * @param string      $path
     * @param boolean     $debug
     */
    public function __construct(ProxyDumper $dumper, Registry $registry, $path, $debug = false)
    {
        $this->dumper   = $dumper;
        $this->dumpers  = [];
        $this->classes  = [];
        $this->path     = $path;
        $this->debug    = $debug;
        $this->registry = $registry;
    }

    /**
     * @return Registry
     */
    public function getRegistry()
    {
        return $this->registry;
    }

    /**
     * @return string
     */
    public function getCachePath()
    {
        return $this->path;
    }

    /**
     * @return boolean
     */
    public function isDebug()
    {
        return $this->debug;
    }

    /**
     * @param array $config
     */
    public function setConfig(array $config)
    {
        foreach ($config['objects'] as $configName => $objectConfig) {
            $className = $objectConfig['object_class'];
            $classConfig = $objectConfig['proxies'];

            $this->register($configName, $className, $classConfig);
        }
    }

    /**
     * @param string $configName
     * @param string $className
     * @param array $classConfig
     *
     * @return Manager
     */
    public function register($configName, $className, $classConfig)
    {
        $config = [];

        foreach ($classConfig as $strategy => $dumpersConfig) {
            foreach ($dumpersConfig as $method => $arguments) {
                $config[sprintf('%s::%s', $className, $method)]= $this->createDumper($strategy, $arguments);
            }
        }

        $config[sprintf('%s::__construct', $className)] = new OrmsConstructorDumper();

        $this->dumpers = array_merge($this->dumpers, $config);
        $this->classes[$configName]= $className;

        return $this;
    }

    /**
     * @return Manager
     */
    public function boot()
    {
        file_put_contents($this->path, $this->getProxiesCode());

        require_once $this->path;

        return $this;
    }

    public function getProxiesCode()
    {
        $this->dumper->setDumpers($this->dumpers);

        $code = '';
        foreach ($this->classes as $configName => $className) {
            $code .= $this->dumper->dump(new \ReflectionClass($className), $configName);
        }

        return '<?php ' . $code;
    }

    /**
     * @param string $strategy
     * @param mixed[] $arguments
     *
     * @return MethodDumperInterface
     */
    private function createDumper($strategy, array $arguments)
    {
        if (!array_key_exists($strategy, $this->strategiesMap)) {
            throw new \InvalidArgumentException(sprintf('The dumping strategy "%s" is not defined.', $strategy));
        }

        $class = $this->strategiesMap[$strategy];

        return new $class(...$arguments);
    }
}
