<?php

namespace Orms\Collections;

use Orms\DataProvider\GenericDataProvider;
use Orms\Proxy\Registry;
use Orms\Statement\Statement;

/**
 * DataObjectCollection return an array Iterable of DataObject
 *
 * @author Xavier HAUSHERR <xavier.hausherr@ebuzzing.com>
 */
class DataObjectCollection implements \Iterator, \Countable, \ArrayAccess
{
    /**
     * Data Provider instance
     * @var \Orms\DataProvider\GenericDataProvider
     */
    protected $provider;

    /**
     * Query Statement
     * @var \Orms\Statement\Statement
     */
    protected $statement;

    /**
     * ClassName to instanciate
     * @var string
     */
    protected $className;

    /**
     * array of links configuration
     * @var array
     */
    protected $links = [];

    /**
     * The registry from where the generated proxies classes call their logic.
     * @var Registry
     */
    private $registry;

    /**
     * Init Iterator
     * @param \Orms\DataProvider\GenericDataProvider $provider
     * @param \Orms\Statement\Statement $statement
     * @param string $className
     * @param array $links
     */
    public function __construct(GenericDataProvider $provider, Statement $statement, $className, array $links)
    {
        $this->provider = $provider;
        $this->statement = $statement;
        $this->className = $className;
        $this->links = $links;
    }

    /**
     * @param Registry $registry
     */
    public function setRegistry(Registry $registry)
    {
        $this->registry = $registry;
    }

    /**
     * @return Registry
     */
    public function getRegistry()
    {
        return $this->registry;
    }

    /**
     * Retourne l'élément courant
     * @return \Orms\Object\DataObject
     */
    public function current()
    {
        $class = $this->getProxyClass($this->className);

        return new $class(
            new \ArrayObject($this->statement->current()),
            $this->registry
        );
    }

    /**
     * Retourne la clé de l'élément courant
     * @return string
     */
    public function key()
    {
        return $this->statement->key();
    }

    /**
     * Se déplace sur l'élément suivant
     */
    public function next()
    {
        $this->statement->next();
    }

    /**
     * Replace l'itérateur sur le premier élément
     */
    public function rewind()
    {
        $this->statement->rewind();
    }

    /**
     * Vérifie si la position courante est valide
     * @return boolean
     */
    public function valid()
    {
        return $this->statement->valid();
    }

    /**
     * Compte le nombre d'éléments
     * @return int
     */
    public function count()
    {
        return $this->statement->count();
    }

    /**
     * Indique si une position existe dans un tableau
     *
     * @param int $offset
     * @return boolean
     */
    public function offsetExists($offset)
    {
        return $offset < $this->count();
    }

    /**
     * Retourne la valeur à la position donnée.
     *
     * @param int $offset
     * @return \Orms\Object\DataObject
     */
    public function offsetGet($offset)
    {
        $class = $this->getProxyClass($this->className);

        return new $class(
            new \ArrayObject($this->statement->fetch(Statement::FETCH_ASSOC, $offset)),
            $this->registry
        );
    }

    /**
     * Assigne une valeur à une position donnée.
     *
     * @deprecated
     * @param int $offset
     * @param mixed $value
     * @throws \LogicException
     */
    public function offsetSet($offset, $value)
    {
        throw new \LogicException(
                'You cannot add an element to a DataObjectCollection');
    }

    /**
     * Supprime une position.
     *
     * @deprecated
     * @param int $offset
     * @throws \LogicException
     */
    public function offsetUnset($offset)
    {
        throw new \LogicException(
                'You cannot unset an element of DataObjectCollection');
    }

    /**
     * @param string $className
     *
     * @return string
     */
    private function getProxyClass($className)
    {
        return '\\Proxies\\'.(new \ReflectionClass($className))->getShortName();
    }
}
