<?php

namespace Overblog\OrmsBundle\Orms;

/**
 * Orms Service
 *
 * @author xavier
 */

use Overblog\OrmsBundle\Orms\ServiceInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Overblog\OrmsBundle\DependencyInjection\OverblogOrmsExtension as Extension;
use Orms\Exception\Exception;

class Service implements ServiceInterface
{
    /**
     * Main service container instance
     * @var ContainerInterface
     */
    protected $container;

    /**
     * Name of the open connections
     * @var array
     */
    protected $connections = array();

    /**
     * Instanciate service
     * @param ContainerInterface $container
     */
    public function __construct(ContainerInterface $container, Array $connections)
    {
        $this->container = $container;
        $this->connections = $connections;
    }

    /**
     * Return Object for config
     * @param string $object
     * @return Overblog\OrmsBundle\Config\Object
     * @throws Exception
     */
    protected function getObjectConfig($object)
    {
        $objectConfig = sprintf(Extension::OBJECT_NAME, $object);

        if($this->getContainer()->has($objectConfig))
        {
            return $this->getContainer()->get($objectConfig);
        }
        else
        {
            throw new Exception("Unknow object requested : " . $object);
        }
    }

    /**
     * Try to retrieve an object instance, or instanciate it from configuration
     * @param string $object
     * @param array $param
     * @return DataObject
     * @throws Exception
     */
    public function getObject($object, $param = null)
    {
        $config = $this->getObjectConfig($object);
        $class  = $config->getClass();

        return new $class(
            $config->getProvider(),
            $param,
            $config->getLinks()
        );
    }

    /**
     * Return an object array based on search param
     * @param string $object
     * @param array $param
     * @return array
     * @throws Exception
     */

    public function getObjects($object, $param = null)
    {
        $config = $this->getObjectConfig($object);
        $class  = $config->getClass();

        return $class::find(
            $config->getProvider(),
            $param,
            $config->getLinks()
        );
    }

    /**
     * Return the first object object array based on search param
     *
     * @param string $object
     * @param array  $param
     *
     * @return array
     *
     * @throws Exception
     */

    public function getFirstObject($object, $param = null)
    {
        $param['offset'] = 0;
        $param['limit']  = 1;

        $os = $this->getObjects($object, $param);

        if (1 === $os->count())
        {
            return $os->offsetGet(0);
        }

        return null;
    }


    /**
     * Return something special based on a DataObject model, but using a self
     * defined special method.
     *
     * @param string $object
     * @param string $method
     * @param array  $param
     *
     * @return mixed
     *
     * @throws Exception
     */

    public function getFromMethod(
        $object,
        $method,
        $param = null
    )
    {
        $config = $this->getObjectConfig($object);
        $class  = $config->getClass();

        return $class::$method(
            $config->getProvider(),
            $param,
            $config->getLinks()
        );
    }

    /**
     * Return an scalar based on search param to count objects
     * @param string $object
     * @param array $param
     * @return int
     * @throws Exception
     */
    public function getNbObjects($object, $param = null)
    {
        try
        {
            $config = $this->getObjectConfig($object);
        }
        catch(Exception $e)
        {
            throw $e;
        }

        $class = $config->getClass();

        return $class::findNb($config->getProvider(), $param);
    }

    /**
     * Return metadata on the last query looking for objects objects
     *
     * @param string $object
     * @param array $param
     * @return int
     * @throws Exception
     */

    public function getMetaData($object)
    {
        $config = $this->getObjectConfig($object);
        $class  = $config->getClass();

        return $class::getMetaData($config->getProvider());
    }

    /**
     * Return container
     * @return ContainerInterface
     */
    public function getContainer()
    {
        return $this->container;
    }

    /**
     * Return ids of connections instanciate
     * @return Array
     */
    public function getConnections()
    {
        return $this->connections;
    }
}
