<?php

namespace Overblog\OrmsBundle\Orms;

/**
 * Orms Service
 *
 * @author xavier
 */

use Orms\Exception\Exception;
use Orms\Object\DataObject;
use Orms\Proxy\Manager;
use Overblog\OrmsBundle\Config\Object;
use Overblog\OrmsBundle\DependencyInjection\OverblogOrmsExtension as Extension;
use Overblog\OrmsBundle\Orms\ServiceInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

class Service implements ServiceInterface
{
    /**
     * Main service container instance
     * @var ContainerInterface
     */

    private $container;

    /**
     * Name of the open connections
     * @var array
     */

    private $connections = [];

    /**
     * Instanciate service
     *
     * @param ContainerInterface $container
     * @param array              $connections
     */

    public function __construct(
        ContainerInterface $container,
        Array $connections
    )
    {
        $this->container   = $container;
        $this->connections = $connections;
    }

    /**
     * Return if object has configuration
     *
     * @param string $object
     *
     * @return bool
     */

    public function hasObjectConfig($object)
    {
        return $this->container->has($this->getObjectConfigName($object));
    }

    /**
     * Return the object configuration name
     *
     * @param $object
     *
     * @return string
     */

    public function getObjectConfigName($object)
    {
        return sprintf(Extension::OBJECT_NAME, $object);
    }

    /**
     * Return Object for config
     *
     * @param string $object
     *
     * @return Object
     * @throws Exception
     */

    private function getObjectConfig($object)
    {
        if (!$this->hasObjectConfig($object))
        {
            throw new Exception(sprintf('Unknown object requested : %s', $object));
        }

        return $this->container->get($this->getObjectConfigName($object));
    }

    /**
     * Try to retrieve an object instance, or instanciate it from configuration
     *
     * @param string $object
     * @param array  $param
     *
     * @return DataObject
     *
     * @throws Exception
     */
    public function getObject($object, $param = NULL)
    {
        $config = $this->getObjectConfig($object);
        $class  = $this->getProxyClass($config->getClass());

        return new $class($param, $this->configureRegistry($config));
    }

    /**
     * Return an object array based on search param
     *
     * @param string $object
     * @param array $param
     *
     * @return \ArrayIterator
     *
     * @throws Exception
     */
    public function getObjects($object, $param = [])
    {
        $config = $this->getObjectConfig($object);
        $class  = $this->getProxyClass($config->getClass());

        $collection = $class::find(
            $config->getProvider(),
            $param,
            $config->getLinks()
        );

        $collection->setRegistry($this->configureRegistry($config));

        return $collection;
    }

    /**
     * Return the first object object array based on search param
     *
     * @param string $object
     * @param array  $param
     *
     * @return DataObject
     *
     * @throws Exception
     */

    public function getFirstObject($object, $param = [])
    {
        // override the offset/limit parameters

        $param['offset'] = 0;
        $param['limit']  = 1;

        $os = $this->getObjects($object, $param);

        if (1 === $os->count())
        {
            return $os->offsetGet(0);
        }

        return null;
    }


    /**
     * Return something special based on a DataObject model, but using a self
     * defined special method.
     *
     * @param string $object
     * @param string $method
     * @param array  $param
     *
     * @return mixed
     *
     * @throws Exception
     */

    public function getFromMethod(
        $object,
        $method,
        $param = []
    )
    {
        $config = $this->getObjectConfig($object);
        $class  = $config->getClass();

        return $class::$method(
            $config->getProvider(),
            $param,
            $config->getLinks()
        );
    }

    /**
     * Returns the object count based on the parameters
     *
     * @param string $object
     * @param array $param
     *
     * @return integer
     *
     * @throws \Exception|\Orms\Exception\Exception
     */

    public function getNbObjects($object, $param = [])
    {
        try
        {
            $config = $this->getObjectConfig($object);
        }
        catch (Exception $e)
        {
            throw $e;
        }

        $class = $config->getClass();

        /** @var DataObject $class */

        return $class::findNb($config->getProvider(), $param);
    }

    /**
     * Return metadata on the last query looking for objects objects
     *
     * @param string $object
     *
     * @return mixed
     *
     * @throws Exception
     */

    public function getMetaData($object)
    {
        $config = $this->getObjectConfig($object);
        $class  = $config->getClass();

        /** @var DataObject $class */

        return $class::getMetaData($config->getProvider());
    }

    /**
     * @param string $object
     */

    public function getManager($object)
    {
        static $manager = [];

        if (! array_key_exists($object, $manager))
        {
            $config = $this->getObjectConfig($object);
            $class = $config->getManager();

            $manager[$object] = new $class($this);
        }

        return $manager[$object];
    }

    /**
     * Return container
     *
     * @return ContainerInterface
     */

    public function getContainer()
    {
        return $this->container;
    }

    /**
     * Return ids of connections instanciate
     *
     * @return Array
     */

    public function getConnections()
    {
        return $this->connections;
    }

    private $proxyManager;

    /**
     * @param Manager $proxyManager
     */
    public function setProxyManager(Manager $proxyManager)
    {
        $this->proxyManager = $proxyManager;
    }

    /**
     * @param string     $configName
     * @param DataObject $entity
     */
    public function attach($configName, DataObject $entity)
    {
        $entity->setRegistry($this->configureRegistry($this->getObjectConfig($configName)));
    }

    /**
     * @param DataObject $entity
     * @param string     $configName
     *
     * @return DataObject
     */
    public function proxify(DataObject $entity, $configName = null)
    {
        $configName = $configName ? $configName : (new \ReflectionClass(get_class($entity)))->getShortName();
        $newEntity  = $this->getObject($configName);
        $newEntity->setFrom($entity);

        $this->attach($configName, $newEntity);

        return $newEntity;
    }

    public function loadProxies()
    {
        $this->proxyManager->boot();
    }

    /**
     * @param  string $className
     *
     * @return string
     */
    private function getProxyClass($className)
    {
        return '\\Proxies\\'.(new \ReflectionClass($className))->getShortName();
    }

    private function configureRegistry($config)
    {
        return $this
            ->proxyManager
            ->getRegistry()
            ->fork()
            ->set('entity.provider', $config->getProvider())
            ->set('entity.links', $config->getLinks())
        ;
    }
}
