<?php

namespace Overblog\OrmsBundle\DependencyInjection;

use Symfony\Component\Config\FileLocator;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Definition;
use Symfony\Component\DependencyInjection\DefinitionDecorator;
use Symfony\Component\DependencyInjection\Loader;
use Symfony\Component\DependencyInjection\Reference;
use Symfony\Component\HttpKernel\DependencyInjection\Extension;

class OverblogOrmsExtension extends Extension
{
    const PROVIDER_NAME = 'orms.provider.%s';
    const CONNECTION_NAME = 'orms.connection.%s';
    const OBJECT_NAME = 'orms.object.config.%s';
    const OBJECT_CLASS = 'orms.object.config.class';

    /**
     * {@inheritDoc}
     */
    public function load(array $configs, ContainerBuilder $container)
    {
        $loader = new Loader\YamlFileLoader($container, new FileLocator(__DIR__.'/../Resources/config'));
        $loader->load('orms.yml');

        $configuration = new Configuration();
        $config = $this->processConfiguration($configuration, $configs);

        // Register connections
        $connections = [];
        foreach ($config['connections'] as $name => $connection) {
            $connections[$name] = sprintf(self::CONNECTION_NAME, $name);
            $this->loadConnection($name, $connection, $container);
        }

        $container->setParameter('orms.connections', $connections);

        // Register providers
        foreach ($config['providers'] as $name => $provider) {
            $this->loadProvider($name, $provider, $container, $config['cache'], $config['links_cache']);
        }

        // Register objects
        foreach ($config['objects'] as $name => $object) {
            $this->loadObject($name, $object, $container);
        }

        $container
            ->getDefinition('orms.proxy_manager')
            ->addMethodCall('setConfig', [$config])
        ;
    }

    /**
     * Loads Orms connection.
     *
     * @param string $name
     * @param array $connection
     * @param ContainerBuilder $container
     */
    private function loadConnection($name, array $connection, ContainerBuilder $container)
    {
        $container
            ->setDefinition(sprintf(self::CONNECTION_NAME, $name), new DefinitionDecorator('orms.connection.abstract'))
            ->setArguments([
                $connection['driver'],
                $connection['options'],
            ])
        ;
    }

    /**
     * Loads Orms provider.
     *
     * @param string $name
     * @param array $provider
     * @param ContainerBuilder $container
     */
    private function loadProvider($name, array $provider, ContainerBuilder $container, $cacheService, $cacheLinksConfig)
    {
        if (null !== $cacheService) {
            $cache = new Reference($cacheService);
        } else {
            $cache = null;
        }

        $container
            ->setDefinition(sprintf(self::PROVIDER_NAME, $name), new Definition($provider['provider_class']))
            ->setArguments([
                new Reference(sprintf(self::CONNECTION_NAME, $provider['connection'])),
                $provider['provider_options'],
                new Reference('service_container'),
                new Reference('orms.logger'),
                $cache,
                new Reference('orms.dispatcher'),
                $cacheLinksConfig,
            ])
        ;
    }

    /**
     * Loads Orms object.
     *
     * @param string $name
     * @param array $object
     * @param ContainerBuilder $container
     */
    private function loadObject($name, array $object, ContainerBuilder $container)
    {
        $container
            ->setDefinition(sprintf(self::OBJECT_NAME, $name), new Definition($container->getParameter(self::OBJECT_CLASS)))
            ->setArguments([
                new Reference('service_container'),
                new Reference(sprintf(self::PROVIDER_NAME, $object['provider'])),
                $object['object_class'],
                $object['links'],
                $object['manager'],
            ])
        ;
    }
}
