<?php

namespace Overblog\OrmsBundle\DependencyInjection;

use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\Config\FileLocator;
use Symfony\Component\HttpKernel\DependencyInjection\Extension;
use Symfony\Component\DependencyInjection\Loader;
use Symfony\Component\DependencyInjection\Definition;
use Symfony\Component\DependencyInjection\Reference;

class OverblogOrmsExtension extends Extension
{
    const PROVIDER_NAME = 'orms.provider.%s';
    const CONNECTION_NAME = 'orms.connection.%s';
    const CONNECTION_CLASS = 'orms.connection.class';
    const OBJECT_NAME = 'orms.object.config.%s';
    const OBJECT_CLASS = 'orms.object.config.class';

    /**
     * {@inheritDoc}
     */
    public function load(array $configs, ContainerBuilder $container)
    {
        $loader = new Loader\YamlFileLoader($container, new FileLocator(__DIR__.'/../Resources/config'));
        $loader->load('orms.yml');

        $configuration = new Configuration();
        $config = $this->processConfiguration($configuration, $configs);

        // Register connections
        foreach($config['connections'] as $name => $connection)
        {
            $this->loadConnection($name, $connection, $container);
        }

        // Register objects
        foreach($config['providers'] as $name => $provider)
        {
            $this->loadProvider($name, $provider, $container, $config['cache'], $config['links_cache']);
        }

        foreach($config['objects'] as $name => $object)
        {
            $this->loadObject($name, $object, $container);
        }

        $container->setParameter('orms.connection', array_keys($container->findTaggedServiceIds('orms.connection')));
    }

    /**
     * Load Orms connections
     * @param string $name
     * @param array $connection
     * @param ContainerBuilder $container
     */
    private function loadConnection($name, Array $connection, ContainerBuilder $container)
    {
        $clientDef = new Definition(
            $container->getParameter(self::CONNECTION_CLASS)
        );

        $clientDef->addArgument($connection['type']);
        $clientDef->addArgument($connection['options']);

        $clientDef->addTag('orms.connection');

        $container->setDefinition(
            sprintf(self::CONNECTION_NAME, $name),
            $clientDef
        );
    }

    /**
     * Load Orms provider
     * @param string $name
     * @param array $provider
     * @param ContainerBuilder $container
     */
    private function loadProvider($name, Array $provider, ContainerBuilder $container, $cacheService, $cacheLinksConfig)
    {
        $clientDef = new Definition($provider['provider_class']);

        if(!is_null($cacheService))
        {
            $cache = new Reference($cacheService);
        }
        else
        {
            $cache = null;
        }

        $clientDef  ->addArgument(new Reference(
                        sprintf(self::CONNECTION_NAME, $provider['connection'])
                    ))
                    ->addArgument($provider['provider_options'])
                    ->addArgument(new Reference('service_container'))
                    ->addArgument(new Reference('orms.logger'))
                    ->AddArgument($cache)
                    ->addArgument(new Reference('orms.dispatcher'))
                    ->AddArgument($cacheLinksConfig);

        $container->setDefinition(
            sprintf(self::PROVIDER_NAME, $name),
            $clientDef
        );
    }

    /**
     * Load Orms object
     *
     * @param string $name
     * @param array $object
     * @param ContainerBuilder $container
     */
    private function loadObject(
                         $name,
        Array            $object,
        ContainerBuilder $container
    )
    {
        $clientDef =
            new Definition(
                $container->getParameter(self::OBJECT_CLASS)
            );

        $clientDef
            ->addArgument(new Reference('service_container'))
            ->addArgument(
                new Reference(
                    sprintf(self::PROVIDER_NAME, $object['provider'])
                )
            )
            ->addArgument($object['object_class'])
            ->addArgument($object['links'])
            ->addArgument($object['manager']);

        $container->setDefinition(
            sprintf(self::OBJECT_NAME, $name),
            $clientDef
        );
    }
}